import { useEffect, useMemo, useState } from "react";
import { Outlet, useNavigate } from "react-router-dom";
import { Layout, Menu, Badge, Dropdown, Switch, Avatar, notification, type MenuProps, Button } from "antd";
import {
  BellOutlined,
  HomeOutlined,
  UserOutlined,
  DollarOutlined,
  CreditCardOutlined,
  FileTextOutlined,
  BarChartOutlined,
  SettingOutlined,
  TeamOutlined,
  DatabaseOutlined,
  FundOutlined,
  NotificationOutlined,
  SwapOutlined
} from "@ant-design/icons";
import { useThemeStore } from "../store/themeStore";
import { useAuthStore } from "../store/authStore";
import { useNotificationStore } from "../store/notificationStore";
import { useQuery, useQueryClient } from "@tanstack/react-query";
import api from "../api/client";
import { can } from "../utils/permissions";
import PlanBadge from "../components/PlanBadge";

const { Header, Sider, Content } = Layout;

const AppLayout = () => {
  const queryClient = useQueryClient();
  const [collapsed, setCollapsed] = useState(false);
  const navigate = useNavigate();
  const { mode, toggle } = useThemeStore();
  const { user, setUser, permissions } = useAuthStore();
  const { unreadCount, setUnreadCount } = useNotificationStore();
  const [apiNotif, contextHolder] = notification.useNotification();
  const [lastNotifId, setLastNotifId] = useState<number | null>(null);

  const unreadQuery = useQuery<number>({
    queryKey: ["notifications", "unread"],
    queryFn: async () => {
      const response = await api.get("/notifications", { params: { unread: true, page_size: 1 } });
      return response.data.data.total as number;
    },
    refetchInterval: 45000
  });

  const latestQuery = useQuery<any>({
    queryKey: ["notifications", "latest"],
    queryFn: async () => {
      const response = await api.get("/notifications", { params: { page: 1, page_size: 1 } });
      return response.data.data.notifications?.[0];
    },
    refetchInterval: 45000
  });

  const activeSelfQuery = useQuery<any>({
    queryKey: ["memberships", "active-self", user?.id],
    enabled: Boolean(user),
    queryFn: async () => (await api.get("/memberships/active-self")).data.data
  });

  const activePlanTypeName = activeSelfQuery.data?.[0]?.plan_type_name as string | undefined;

  useEffect(() => {
    if (typeof unreadQuery.data === "number") setUnreadCount(unreadQuery.data);
  }, [unreadQuery.data, setUnreadCount]);

  useEffect(() => {
    const latest = latestQuery.data;
    if (latest?.id && latest.id !== lastNotifId && !latest.read_at) {
      setLastNotifId(latest.id);
      apiNotif.open({
        message: latest.title,
        description: latest.message,
        placement: "bottomRight"
      });
    }
  }, [latestQuery.data, apiNotif, lastNotifId, setLastNotifId]);

  const menuItems: MenuProps["items"] = useMemo(() => {
    const base: NonNullable<MenuProps["items"]> = [
      ...(can(permissions, "home.view") ? [{ key: "/app/home", icon: <HomeOutlined />, label: "Home" }] : []),
      ...(can(permissions, "profile.view") ? [{ key: "/app/profile", icon: <UserOutlined />, label: "Profile" }] : []),
      ...(can(permissions, "credits.view") ? [{ key: "/app/credits", icon: <DollarOutlined />, label: "Credits" }] : []),
      ...(can(permissions, "debits.view") ? [{ key: "/app/debits", icon: <CreditCardOutlined />, label: "Debits" }] : []),
      ...(can(permissions, "requests.submit") ? [{ key: "/app/requests", icon: <FileTextOutlined />, label: "Requests" }] : []),
      ...(can(permissions, "requests.view_all")
        ? [{ key: "/app/requests-view", icon: <FileTextOutlined />, label: "All Requests" }]
        : []),
      ...(can(permissions, "notifications.view")
        ? [{ key: "/app/notifications", icon: <NotificationOutlined />, label: "Notifications" }]
        : []),
      ...(can(permissions, "settings.view") ? [{ key: "/app/settings", icon: <SettingOutlined />, label: "Settings" }] : [])
    ];

    if (can(permissions, "reports.view")) {
      const insertAt = Math.min(5, base.length);
      base.splice(insertAt, 0, { key: "/app/reports", icon: <BarChartOutlined />, label: "Reports" });
    }

    if (can(permissions, "admin.overview.view") || can(permissions, "admin.member_savings.view") || can(permissions, "admin.cashier_analysis.view")) {
      base.push({ key: "admin-divider", label: "Admin", disabled: true } as any);
      if (can(permissions, "admin.overview.view")) base.push({ key: "/app/admin/overview", icon: <BarChartOutlined />, label: "Overview" });
      if (user?.access_level === "administrator") {
        base.push({ key: "/app/admin/users", icon: <TeamOutlined />, label: "Users" });
        base.push({ key: "/app/admin/members", icon: <TeamOutlined />, label: "Members" });
        base.push({ key: "/app/admin/cashiers", icon: <SwapOutlined />, label: "Cashiers" });
        base.push({ key: "/app/admin/savings-plan-types", icon: <FundOutlined />, label: "Savings Plans" });
        base.push({ key: "/app/admin/charity-types", icon: <FundOutlined />, label: "Charity Types" });
        base.push({ key: "/app/admin/loan-requests", icon: <FileTextOutlined />, label: "Loan Requests" });
        base.push({ key: "/app/admin/charity-requests", icon: <FileTextOutlined />, label: "Charity Requests" });
        base.push({ key: "/app/admin/release-requests", icon: <FileTextOutlined />, label: "Release Requests" });
      }
      if (can(permissions, "admin.member_savings.view")) base.push({ key: "/app/admin/member-savings", icon: <DollarOutlined />, label: "Member Savings" });
      if (can(permissions, "admin.cashier_analysis.view"))
        base.push({ key: "/app/admin/cashier-analysis", icon: <BarChartOutlined />, label: "Cashier Analysis" });
      if (user?.access_level === "administrator") {
        base.push({ key: "/app/admin/activity", icon: <SettingOutlined />, label: "Activity" });
        base.push({ key: "/app/admin/excel", icon: <DatabaseOutlined />, label: "Excel Import/Export" });
      }
    }

    return base as MenuProps["items"];
  }, [user, permissions]);

  const userMenu = {
    items: [
      { key: "profile", label: "Profile", onClick: () => navigate("/app/profile") },
      { key: "sessions", label: "Sessions", onClick: () => navigate("/app/profile") },
      {
        key: "logout",
        label: "Logout",
        onClick: async () => {
          try {
            await api.post("/auth/logout");
          } catch {
            // ignore
          }
          queryClient.clear();
          setUnreadCount(0);
          setUser(undefined);
          navigate("/login");
        }
      }
    ]
  };

  return (
    <Layout className="erp-shell">
      {contextHolder}
      <Sider
        collapsible
        collapsed={collapsed}
        onCollapse={setCollapsed}
        className="erp-sider"
        breakpoint="lg"
        collapsedWidth={0}
        trigger={null}
      >
        <div style={{ padding: 20, color: "var(--app-text)", fontWeight: 700, letterSpacing: 1 }}>
          {collapsed ? "J" : "Jana ERP"}
        </div>
        <Menu
          theme={mode === "dark" ? "dark" : "light"}
          mode="inline"
          items={menuItems}
          onClick={({ key }) => {
            if (key.startsWith("/")) {
              navigate(key);
              if (window.innerWidth < 992) setCollapsed(true);
            }
          }}
        />
      </Sider>
      <Layout>
        <Header className="erp-header">
          <div style={{ display: "flex", alignItems: "center", gap: 12 }}>
            <Button
              type="text"
              onClick={() => setCollapsed((v) => !v)}
              style={{ color: "var(--app-text)", padding: 0, width: 36 }}
              aria-label="Toggle menu"
            >
              ☰
            </Button>
            <span style={{ fontWeight: 600 }}>Savings & Loan ERP</span>
          </div>
          <div style={{ display: "flex", alignItems: "center", gap: 16 }}>
            <Switch checked={mode === "dark"} onChange={toggle} checkedChildren="Dark" unCheckedChildren="Light" />
            {activePlanTypeName && <PlanBadge planTypeName={activePlanTypeName} size={28} />}
            <Badge count={unreadCount} size="small">
              <BellOutlined style={{ fontSize: 18 }} onClick={() => navigate("/app/notifications")} />
            </Badge>
            <Dropdown menu={userMenu} placement="bottomRight">
              <div style={{ display: "flex", alignItems: "center", gap: 8, cursor: "pointer" }}>
                <Avatar icon={<UserOutlined />} />
                <span>{user?.full_name || "User"}</span>
              </div>
            </Dropdown>
          </div>
        </Header>
        <Content className="erp-content">
          <Outlet />
        </Content>
      </Layout>
    </Layout>
  );
};

export default AppLayout;
