import { useEffect, useMemo, useState } from "react";
import { Card, Table, Input, Button, Drawer, Form, Select, message, Space, Tag, DatePicker } from "antd";
import { DownloadOutlined, PlusOutlined } from "@ant-design/icons";
import { useQuery, useQueryClient } from "@tanstack/react-query";
import dayjs from "dayjs";
import api from "../api/client";
import { useAuthStore } from "../store/authStore";
import { can } from "../utils/permissions";

interface MemberInfo {
  member_id: string;
  member_short_name: string;
  plan_type_name?: string;
  monthly_saving_amount?: number;
  charity_amount?: number;
}

const CreditsPage = () => {
  const queryClient = useQueryClient();
  const { user, permissions } = useAuthStore();
  const [searchInput, setSearchInput] = useState("");
  const [search, setSearch] = useState("");
  const [typeFilter, setTypeFilter] = useState<string | undefined>();
  const [page, setPage] = useState(1);
  const [pageSize, setPageSize] = useState(20);
  const [sortField, setSortField] = useState<string>("created_at");
  const [sortOrder, setSortOrder] = useState<"asc" | "desc">("desc");
  const [open, setOpen] = useState(false);
  const [editing, setEditing] = useState<any | null>(null);
  const [form] = Form.useForm();
  const [memberInfo, setMemberInfo] = useState<MemberInfo | null>(null);
  const [loanOptions, setLoanOptions] = useState<any[]>([]);
  const [outstanding, setOutstanding] = useState<number | null>(null);
  const [remaining, setRemaining] = useState<number | null>(null);
  const [amountReadOnly, setAmountReadOnly] = useState(false);
  const [transactionType, setTransactionType] = useState<string | null>(null);

  const canCreate = can(permissions, "credits.create");
  const canUpdate = can(permissions, "credits.update");
  const canDelete = can(permissions, "credits.delete");
  const isAdminOrEditor = canCreate || canUpdate || canDelete;
  const listUrl = "/credits";

  useEffect(() => {
    const handle = setTimeout(() => setSearch(searchInput), 300);
    return () => clearTimeout(handle);
  }, [searchInput]);

  useEffect(() => {
    setPage(1);
  }, [search, typeFilter, listUrl]);

  const { data, isLoading } = useQuery({
    queryKey: ["credits", listUrl, search, typeFilter, page, pageSize, sortField, sortOrder],
    queryFn: async () => {
      const response = await api.get(listUrl, {
        params: {
          page,
          page_size: pageSize,
          search,
          transaction_type: typeFilter,
          sort: sortField,
          order: sortOrder
        }
      });
      return response.data.data;
    }
  });

  const membersQuery = useQuery({
    queryKey: ["memberships-active"],
    enabled: canCreate || canUpdate,
    queryFn: async () => {
      const response = await api.get("/memberships/active");
      return response.data.data;
    }
  });

  const summaryCards = useMemo(() => {
    return data?.summary || {};
  }, [data]);

  const onCreate = async (values: any) => {
    try {
      const payload = {
        ...values,
        transaction_date: values?.transaction_date?.format
          ? values.transaction_date.format("YYYY-MM-DD")
          : values.transaction_date
      };
      if (editing) {
        await api.put(`/credits/${editing.id}`, payload);
        message.success("Credit updated");
      } else {
        await api.post("/credits", payload);
        message.success("Credit created");
      }
      setOpen(false);
      setEditing(null);
      form.resetFields();
      setMemberInfo(null);
      setLoanOptions([]);
      setOutstanding(null);
      setRemaining(null);
      setTransactionType(null);
      setAmountReadOnly(false);
      queryClient.invalidateQueries({ queryKey: ["credits"] });
    } catch (err: any) {
      message.error(err?.response?.data?.message || "Save failed");
    }
  };

  const onEdit = (record: any) => {
    setEditing(record);
    setOpen(true);
    setTransactionType(record.transaction_type);
    const member = (membersQuery.data || []).find((m: any) => m.member_id === record.member_id);
    setMemberInfo(member || null);
    form.setFieldsValue({
      member_id: record.member_id,
      transaction_type: record.transaction_type,
      total_amount: record.total_amount,
      savings_amount: record.savings_amount,
      charity_amount: record.charity_amount,
      loan_id: record.loan_id,
      transaction_date: record.transaction_date ? dayjs(record.transaction_date) : undefined
    });
  };

  const onDelete = async (record: any) => {
    try {
      await api.delete(`/credits/${record.id}`);
      message.success("Credit deleted");
      queryClient.invalidateQueries({ queryKey: ["credits"] });
    } catch (err: any) {
      message.error(err?.response?.data?.message || "Delete failed");
    }
  };

  const exportCsv = () => {
    const rows = (data?.credits || []).map((row: any) => ({
      transaction_id: row.transaction_id,
      member_id: row.member_id,
      member_short_name: row.member_short_name,
      transaction_type: row.transaction_type,
      total_amount: row.total_amount,
      transaction_date: row.transaction_date
    }));
    if (!rows.length) return;
    const header = Object.keys(rows[0]).join(",");
    const body = rows.map((r: any) => Object.values(r).join(",")).join("\n");
    const csv = [header, body].join("\n");
    const blob = new Blob([csv], { type: "text/csv;charset=utf-8;" });
    const url = URL.createObjectURL(blob);
    const link = document.createElement("a");
    link.href = url;
    link.setAttribute("download", "credits.csv");
    document.body.appendChild(link);
    link.click();
    link.remove();
  };

  const handleMemberChange = async (value: string) => {
    const member = (membersQuery.data || []).find((m: any) => m.member_id === value);
    setMemberInfo(member || null);

    setTransactionType(null);
    setLoanOptions([]);
    setOutstanding(null);
    setRemaining(null);
    setAmountReadOnly(false);
    form.setFieldsValue({
      transaction_type: undefined,
      loan_id: undefined,
      total_amount: undefined,
      savings_amount: 0,
      charity_amount: 0
    });
  };

  const loadLoans = async (memberId: string) => {
    try {
      const response = await api.get(`/loans/member/${memberId}/active`);
      const loans = response.data.data.loans || [];
      setLoanOptions(loans);
      if (!loans.length) {
        message.warning("This member has no active loans");
      }
    } catch (err: any) {
      message.error("Failed to load loans");
    }
  };

  const handleTransactionTypeChange = async (value: string) => {
    setTransactionType(value);
    const memberId = form.getFieldValue("member_id");
    if (!memberId) {
      message.warning("Select a member first");
      form.setFieldsValue({ transaction_type: undefined });
      setTransactionType(null);
      return;
    }

    if (value === "savings" && memberInfo) {
      const total = (memberInfo.monthly_saving_amount || 0) + (memberInfo.charity_amount || 0);
      setAmountReadOnly(true);
      form.setFieldsValue({
        total_amount: total.toFixed(2),
        savings_amount: memberInfo.monthly_saving_amount || 0,
        charity_amount: memberInfo.charity_amount || 0
      });
    } else {
      setAmountReadOnly(false);
      form.setFieldsValue({ total_amount: undefined, savings_amount: 0, charity_amount: 0 });
    }

    if (value === "emi") {
      await loadLoans(memberId);
    } else {
      setLoanOptions([]);
      setOutstanding(null);
      setRemaining(null);
      form.setFieldsValue({ loan_id: undefined });
    }
  };

  const handleAmountChange = (value: string) => {
    const transactionType = form.getFieldValue("transaction_type");
    const amount = Number(value || 0);

    if (transactionType === "flexi") {
      form.setFieldsValue({ savings_amount: amount, charity_amount: 0 });
    }

    if (transactionType === "charity") {
      form.setFieldsValue({ savings_amount: 0, charity_amount: amount });
    }

    if (transactionType === "emi") {
      const remainingAmount = (outstanding || 0) - amount;
      setRemaining(remainingAmount > 0 ? remainingAmount : 0);
      form.setFieldsValue({ savings_amount: 0, charity_amount: 0 });
    }
  };

  const handleLoanChange = (value: string) => {
    const loan = loanOptions.find((item) => item.loan_id === value);
    if (loan) {
      setOutstanding(loan.outstanding);
      setRemaining(loan.outstanding);
    } else {
      setOutstanding(null);
      setRemaining(null);
    }
  };

  const columns = [
    { title: "Transaction ID", dataIndex: "transaction_id", width: 120 },
    { title: "Member ID", dataIndex: "member_id", width: 110 },
    { title: "Name", dataIndex: "member_short_name", width: 140 },
    {
      title: "Type",
      dataIndex: "transaction_type",
      width: 90,
      render: (value: string) => <Tag color="blue">{value}</Tag>
    },
    { title: "Amount", dataIndex: "total_amount", width: 110 },
    { title: "Savings", dataIndex: "savings_amount", width: 110 },
    { title: "Charity", dataIndex: "charity_amount", width: 110 },
    { title: "Date", dataIndex: "transaction_date", width: 110 },
    { title: "Created", dataIndex: "created_at", width: 140 },
    { title: "Cashier", dataIndex: "cashier_name", width: 140 },
    canUpdate || canDelete
      ? {
          title: "Actions",
          width: 140,
          render: (_: any, record: any) => (
            <Space>
              <Button size="small" onClick={() => onEdit(record)} disabled={!canUpdate}>
                Edit
              </Button>
              <Button size="small" danger onClick={() => onDelete(record)} disabled={!canDelete}>
                Delete
              </Button>
            </Space>
          )
        }
      : {}
  ].filter(Boolean) as any[];

  return (
    <div>
      <Card className="panel" style={{ marginBottom: 16 }}>
        <Space size="large" wrap>
          <Card className="panel" style={{ minWidth: 180 }}>
            <div>Total Amount</div>
            <div style={{ fontSize: 18 }}>{summaryCards.totalAmount || 0}</div>
          </Card>
          <Card className="panel" style={{ minWidth: 180 }}>
            <div>Savings Amount</div>
            <div style={{ fontSize: 18 }}>{summaryCards.savingsAmount || 0}</div>
          </Card>
          <Card className="panel" style={{ minWidth: 180 }}>
            <div>EMI Amount</div>
            <div style={{ fontSize: 18 }}>{summaryCards.emiAmount || 0}</div>
          </Card>
          <Card className="panel" style={{ minWidth: 180 }}>
            <div>Charity Amount</div>
            <div style={{ fontSize: 18 }}>{summaryCards.charityAmount || 0}</div>
          </Card>
        </Space>
      </Card>

      <Card
        className="panel"
        title="Credits"
        extra={
          <Space>
            <Input
              placeholder="Search credits"
              value={searchInput}
              onChange={(e) => setSearchInput(e.target.value)}
              allowClear
            />
            <Select
              allowClear
              placeholder="Filter Type"
              style={{ width: 160 }}
              onChange={(value) => setTypeFilter(value)}
              options={[
                { label: "Savings", value: "savings" },
                { label: "EMI", value: "emi" },
                { label: "Charity", value: "charity" }
              ]}
            />
            {isAdminOrEditor && (
              <Button icon={<DownloadOutlined />} onClick={exportCsv}>
                Export
              </Button>
            )}
            {canCreate && (
              <Button
                type="primary"
                icon={<PlusOutlined />}
                onClick={() => {
                  setEditing(null);
                  form.resetFields();
                  form.setFieldsValue({ transaction_date: dayjs() });
                  setMemberInfo(null);
                  setLoanOptions([]);
                  setOutstanding(null);
                  setRemaining(null);
                  setTransactionType(null);
                  setOpen(true);
                }}
              >
                New Credit
              </Button>
            )}
          </Space>
        }
      >
        <Table
          columns={columns}
          dataSource={data?.credits || []}
          loading={isLoading}
          rowKey="id"
          pagination={{
            current: page,
            pageSize,
            total: data?.total || 0,
            showSizeChanger: true
          }}
          scroll={{ x: 1200 }}
          onChange={(pagination, _filters, sorter: any) => {
            setPage(pagination.current || 1);
            setPageSize(pagination.pageSize || 20);
            if (sorter?.field) {
              setSortField(sorter.field);
              setSortOrder(sorter.order === "ascend" ? "asc" : "desc");
            }
          }}
        />
      </Card>

      <Drawer
        title={editing ? "Edit Credit" : "Create Credit"}
        open={open}
        placement="right"
        onClose={() => setOpen(false)}
        width={420}
      >
        <Form form={form} layout="vertical" onFinish={onCreate}>
          <Form.Item name="member_id" label="Member" rules={[{ required: true }]}>
            <Select
              options={(membersQuery.data || []).map((m: any) => ({
                label: `${m.member_id} - ${m.member_short_name}`,
                value: m.member_id
              }))}
              onChange={handleMemberChange}
              disabled={!canCreate && !editing}
            />
          </Form.Item>

          <Card className="panel" style={{ marginBottom: 12 }}>
            <div>Plan Type: {memberInfo?.plan_type_name || "-"}</div>
            <div>Monthly Savings: {memberInfo?.monthly_saving_amount || 0}</div>
            <div>Charity Amount: {memberInfo?.charity_amount || 0}</div>
          </Card>

          <Form.Item name="transaction_type" label="Type" rules={[{ required: true }]}>
            <Select
              onChange={handleTransactionTypeChange}
              options={[
                { label: "Savings", value: "savings" },
                { label: "Flexi Savings", value: "flexi" },
                { label: "EMI", value: "emi" },
                { label: "Charity", value: "charity" }
              ]}
            />
          </Form.Item>

          {transactionType === "emi" && (
            <>
              <Form.Item name="loan_id" label="Loan ID" rules={[{ required: true }]}>
                <Select
                  onChange={handleLoanChange}
                  options={loanOptions.map((loan) => ({
                    label: `${loan.loan_id} - Outstanding ${loan.outstanding.toFixed(2)}`,
                    value: loan.loan_id
                  }))}
                />
              </Form.Item>
              <Card className="panel" style={{ marginBottom: 12 }}>
                <div>Outstanding: {outstanding?.toFixed(2) || "-"}</div>
                <div>Remaining: {remaining?.toFixed(2) || "-"}</div>
              </Card>
            </>
          )}

          <Form.Item name="total_amount" label="Total Amount" rules={[{ required: true }]}>
            <Input
              placeholder={
                transactionType === "flexi"
                  ? "Enter flexi savings amount"
                  : transactionType === "emi"
                    ? "Enter EMI payment amount"
                    : transactionType === "charity"
                      ? "Enter charity amount"
                      : undefined
              }
              onChange={(e) => handleAmountChange(e.target.value)}
              readOnly={amountReadOnly}
            />
          </Form.Item>

          <Form.Item name="transaction_date" label="Transaction Date" rules={[{ required: true }]}>
            <DatePicker style={{ width: "100%" }} format="YYYY-MM-DD" />
          </Form.Item>

          <Form.Item name="savings_amount" hidden>
            <Input />
          </Form.Item>
          <Form.Item name="charity_amount" hidden>
            <Input />
          </Form.Item>

          <Button type="primary" htmlType="submit" block>
            Save
          </Button>
        </Form>
      </Drawer>
    </div>
  );
};

export default CreditsPage;
