import { Card, Row, Col, Statistic, Tooltip, Space } from "antd";
import { DollarOutlined, SaveOutlined, FundOutlined, CheckCircleOutlined, InfoCircleOutlined } from "@ant-design/icons";
import { useQuery } from "@tanstack/react-query";
import ReactECharts from "echarts-for-react";
import api from "../api/client";

const Dashboard = () => {
  const { data } = useQuery({
    queryKey: ["overall-summary"],
    queryFn: async () => {
      const response = await api.get("/reports/overall-summary");
      return response.data.data;
    }
  });

  const { data: monthly } = useQuery({
    queryKey: ["monthly-trends"],
    queryFn: async () => {
      const response = await api.get("/reports/monthly-trends");
      return response.data.data;
    }
  });

  const { data: charity } = useQuery({
    queryKey: ["charity-analysis"],
    queryFn: async () => {
      const response = await api.get("/reports/charity-analysis");
      return response.data.data;
    }
  });

  const summary = data || {};

  const chartOption = {
    tooltip: { trigger: "axis" },
    legend: { textStyle: { color: "#e2e8f0" } },
    xAxis: { type: "category", data: monthly?.months || [], axisLabel: { color: "#94a3b8" } },
    yAxis: { type: "value", axisLabel: { color: "#94a3b8" } },
    series: [
      { name: "Savings", type: "line", data: monthly?.savings || [] },
      { name: "Charity", type: "line", data: monthly?.charity || [] },
      { name: "Loan", type: "line", data: monthly?.loan || [] },
      { name: "EMI", type: "line", data: monthly?.emi || [] }
    ]
  };

  const charityOption = {
    tooltip: { trigger: "axis" },
    xAxis: {
      type: "category",
      data: (charity || []).map((item: any) => item.type),
      axisLabel: { color: "#94a3b8", rotate: 20 }
    },
    yAxis: { type: "value", axisLabel: { color: "#94a3b8" } },
    series: [
      {
        name: "Allotted Amount",
        type: "bar",
        barWidth: "55%",
        data: (charity || []).map((item: any) => Number(item.allotted_amount || 0)),
        itemStyle: { color: "#1677ff" }
      }
    ]
  };

  return (
    <div>
      <Row gutter={[16, 16]}>
        <Col xs={24} md={6}>
          <Card className="panel">
            <Statistic
              title={
                <Space size={6}>
                  <DollarOutlined style={{ color: "#4cc9f0" }} />
                  Total Credits
                  <Tooltip title="Sum of all credits (savings, charity, EMI)">
                    <InfoCircleOutlined style={{ color: "#94a3b8" }} />
                  </Tooltip>
                </Space>
              }
              value={summary.total_amount || 0}
            />
          </Card>
        </Col>
        <Col xs={24} md={6}>
          <Card className="panel">
            <Statistic
              title={
                <Space size={6}>
                  <SaveOutlined style={{ color: "#4895ef" }} />
                  Total Savings
                  <Tooltip title="Total savings credits (including flexi stored as savings)">
                    <InfoCircleOutlined style={{ color: "#94a3b8" }} />
                  </Tooltip>
                </Space>
              }
              value={summary.total_savings || 0}
            />
          </Card>
        </Col>
        <Col xs={24} md={6}>
          <Card className="panel">
            <Statistic
              title={
                <Space size={6}>
                  <FundOutlined style={{ color: "#f72585" }} />
                  Total Loans
                  <Tooltip title="Total loans disbursed (released)">
                    <InfoCircleOutlined style={{ color: "#94a3b8" }} />
                  </Tooltip>
                </Space>
              }
              value={summary.total_borrowed || 0}
            />
          </Card>
        </Col>
        <Col xs={24} md={6}>
          <Card className="panel">
            <Statistic
              title={
                <Space size={6}>
                  <CheckCircleOutlined style={{ color: "#2ec4b6" }} />
                  Loan Paid
                  <Tooltip title="Total EMI credits posted">
                    <InfoCircleOutlined style={{ color: "#94a3b8" }} />
                  </Tooltip>
                </Space>
              }
              value={summary.total_loan_paid || 0}
            />
          </Card>
        </Col>
      </Row>
      <Row gutter={[16, 16]} style={{ marginTop: 16 }}>
        <Col xs={24} md={16}>
          <Card className="panel" title="Monthly Trends">
            <ReactECharts option={chartOption} style={{ height: 320 }} />
          </Card>
        </Col>
        <Col xs={24} md={8}>
          <Card className="panel" title="Charity Distribution">
            <ReactECharts option={charityOption} style={{ height: 320 }} />
          </Card>
        </Col>
      </Row>
    </div>
  );
};

export default Dashboard;
