import { useEffect, useMemo, useState } from "react";
import { Card, Table, Input, Button, Drawer, Form, Select, message, Space, Tag } from "antd";
import { DownloadOutlined } from "@ant-design/icons";
import { useQuery, useQueryClient } from "@tanstack/react-query";
import api from "../api/client";
import { useAuthStore } from "../store/authStore";
import { can } from "../utils/permissions";

const DebitsPage = () => {
  const queryClient = useQueryClient();
  const { user, permissions } = useAuthStore();
  const [searchInput, setSearchInput] = useState("");
  const [search, setSearch] = useState("");
  const [typeFilter, setTypeFilter] = useState<string | undefined>();
  const [page, setPage] = useState(1);
  const [pageSize, setPageSize] = useState(20);
  const [sortField, setSortField] = useState<string>("transaction_date");
  const [sortOrder, setSortOrder] = useState<"asc" | "desc">("desc");
  const [open, setOpen] = useState(false);
  const [editing, setEditing] = useState<any | null>(null);
  const [form] = Form.useForm();

  const canUpdate = can(permissions, "debits.update");
  const canDelete = can(permissions, "debits.delete");
  const isAdminOrEditor = canUpdate || canDelete;
  const listUrl = "/debits";

  useEffect(() => {
    const handle = setTimeout(() => setSearch(searchInput), 300);
    return () => clearTimeout(handle);
  }, [searchInput]);

  useEffect(() => {
    setPage(1);
  }, [search, typeFilter, listUrl]);

  const { data, isLoading } = useQuery({
    queryKey: ["debits", listUrl, search, typeFilter, page, pageSize, sortField, sortOrder],
    queryFn: async () => {
      const response = await api.get(listUrl, {
        params: {
          page,
          page_size: pageSize,
          search,
          transaction_type: typeFilter,
          sort: sortField,
          order: sortOrder
        }
      });
      return response.data.data;
    }
  });

  const summary = useMemo(() => data?.summary || {}, [data]);

  const onEdit = (record: any) => {
    setEditing(record);
    setOpen(true);
    form.setFieldsValue({
      member_id: record.member_id,
      transaction_type: record.transaction_type,
      amount: record.amount,
      cashier_id: record.cashier_id,
      transaction_date: record.transaction_date,
      notes: record.notes
    });
  };

  const onSave = async (values: any) => {
    try {
      await api.put(`/debits/${editing.id}`, values);
      message.success("Debit updated");
      setOpen(false);
      setEditing(null);
      form.resetFields();
      queryClient.invalidateQueries({ queryKey: ["debits"] });
    } catch (err: any) {
      message.error(err?.response?.data?.message || "Save failed");
    }
  };

  const onDelete = async (record: any) => {
    try {
      await api.delete(`/debits/${record.id}`);
      message.success("Debit deleted");
      queryClient.invalidateQueries({ queryKey: ["debits"] });
    } catch (err: any) {
      message.error(err?.response?.data?.message || "Delete failed");
    }
  };

  const exportCsv = () => {
    const rows = (data?.debits || []).map((row: any) => ({
      transaction_id: row.transaction_id,
      member_id: row.member_id,
      transaction_type: row.transaction_type,
      amount: row.amount,
      transaction_date: row.transaction_date
    }));
    if (!rows.length) return;
    const header = Object.keys(rows[0]).join(",");
    const body = rows.map((r: any) => Object.values(r).join(",")).join("\n");
    const csv = [header, body].join("\n");
    const blob = new Blob([csv], { type: "text/csv;charset=utf-8;" });
    const url = URL.createObjectURL(blob);
    const link = document.createElement("a");
    link.href = url;
    link.setAttribute("download", "debits.csv");
    document.body.appendChild(link);
    link.click();
    link.remove();
  };

  const columns = [
    { title: "Transaction ID", dataIndex: "transaction_id", width: 140 },
    { title: "Member ID", dataIndex: "member_id", width: 120 },
    { title: "Name", dataIndex: "member_short_name", width: 160 },
    {
      title: "Type",
      dataIndex: "transaction_type",
      width: 140,
      render: (value: string) => <Tag color="purple">{value}</Tag>
    },
    { title: "Amount", dataIndex: "amount", width: 120 },
    { title: "Cashier", dataIndex: "cashier_name", width: 160 },
    { title: "Date", dataIndex: "transaction_date", width: 120 },
    { title: "Notes", dataIndex: "notes", ellipsis: true },
    isAdminOrEditor
      ? {
          title: "Actions",
          width: 140,
          render: (_: any, record: any) => (
            <Space>
              <Button size="small" onClick={() => onEdit(record)} disabled={!canUpdate}>
                Edit
              </Button>
              <Button size="small" danger onClick={() => onDelete(record)} disabled={!canDelete}>
                Delete
              </Button>
            </Space>
          )
        }
      : {}
  ].filter(Boolean) as any[];

  return (
    <div>
      <Card className="panel" style={{ marginBottom: 16 }}>
        <Space size="large" wrap>
          <Card className="panel" style={{ minWidth: 200 }}>
            <div>Total Debit Amount</div>
            <div style={{ fontSize: 18 }}>{summary.totalAmount || 0}</div>
          </Card>
          <Card className="panel" style={{ minWidth: 200 }}>
            <div>Loan Disbursement</div>
            <div style={{ fontSize: 18 }}>{summary.loanDisbursementAmount || 0}</div>
          </Card>
          <Card className="panel" style={{ minWidth: 200 }}>
            <div>Savings Release</div>
            <div style={{ fontSize: 18 }}>{summary.savingsReleaseAmount || 0}</div>
          </Card>
          <Card className="panel" style={{ minWidth: 200 }}>
            <div>Charity</div>
            <div style={{ fontSize: 18 }}>{summary.charityAmount || 0}</div>
          </Card>
        </Space>
      </Card>

      <Card
        className="panel"
        title="Debits"
        extra={
          <Space>
            <Input
              placeholder="Search debits"
              value={searchInput}
              onChange={(e) => setSearchInput(e.target.value)}
              allowClear
            />
            <Select
              allowClear
              placeholder="Filter Type"
              style={{ width: 160 }}
              onChange={(value) => setTypeFilter(value)}
              options={[
                { label: "Loan Disbursement", value: "loan_disbursement" },
                { label: "Charity", value: "charity" },
                { label: "Savings Release", value: "savings_release" }
              ]}
            />
            {isAdminOrEditor && (
              <Button icon={<DownloadOutlined />} onClick={exportCsv}>
                Export
              </Button>
            )}
          </Space>
        }
      >
        <Table
          columns={columns}
          dataSource={data?.debits || []}
          loading={isLoading}
          rowKey="id"
          pagination={{
            current: page,
            pageSize,
            total: data?.total || 0,
            showSizeChanger: true
          }}
          scroll={{ x: 1100 }}
          onChange={(pagination, _filters, sorter: any) => {
            setPage(pagination.current || 1);
            setPageSize(pagination.pageSize || 20);
            if (sorter?.field) {
              setSortField(sorter.field);
              setSortOrder(sorter.order === "ascend" ? "asc" : "desc");
            }
          }}
        />
      </Card>

      <Drawer
        title="Edit Debit"
        open={open}
        placement="right"
        onClose={() => setOpen(false)}
        width={420}
      >
        <Form form={form} layout="vertical" onFinish={onSave}>
          <Form.Item name="member_id" label="Member" rules={[{ required: true }]}>
            <Input />
          </Form.Item>
          <Form.Item name="transaction_type" label="Type" rules={[{ required: true }]}>
            <Select
              options={[
                { label: "Loan Disbursement", value: "loan_disbursement" },
                { label: "Charity", value: "charity" },
                { label: "Savings Release", value: "savings_release" }
              ]}
            />
          </Form.Item>
          <Form.Item name="amount" label="Amount" rules={[{ required: true }]}>
            <Input />
          </Form.Item>
          <Form.Item name="cashier_id" label="Cashier ID" rules={[{ required: true }]}>
            <Input />
          </Form.Item>
          <Form.Item name="transaction_date" label="Transaction Date" rules={[{ required: true }]}>
            <Input placeholder="YYYY-MM-DD" />
          </Form.Item>
          <Form.Item name="notes" label="Notes">
            <Input />
          </Form.Item>
          <Button type="primary" htmlType="submit" block>
            Save
          </Button>
        </Form>
      </Drawer>
    </div>
  );
};

export default DebitsPage;
