import { useMemo, useState } from "react";
import { Card, Row, Col, Statistic, Table, Segmented, Typography, Space, Tooltip } from "antd";
import { InfoCircleOutlined, TeamOutlined, UserDeleteOutlined, FundOutlined, HeartOutlined } from "@ant-design/icons";
import { useQuery } from "@tanstack/react-query";
import ReactECharts from "echarts-for-react";
import api from "../api/client";
import { useAuthStore } from "../store/authStore";
import { can } from "../utils/permissions";

const { Text } = Typography;

const formatCurrency = (value: number) =>
  new Intl.NumberFormat("en-IN", { minimumFractionDigits: 2, maximumFractionDigits: 2 }).format(value || 0);

const ReportsPage = () => {
  const { user, permissions } = useAuthStore();
  const [savingsLoanView, setSavingsLoanView] = useState<"overall" | "active">("overall");
  const canReports = can(permissions, "reports.view");
  const canCashier = can(permissions, "cashier_summary.view");

  const { data: counts } = useQuery({
    queryKey: ["report-counts"],
    queryFn: async () => (await api.get("/reports/summary-counts")).data.data
    ,
    enabled: Boolean(user) && canReports
  });

  const { data: summary } = useQuery({
    queryKey: ["overall-summary"],
    queryFn: async () => (await api.get("/reports/overall-summary")).data.data,
    enabled: Boolean(user) && canReports
  });

  const { data: activeSummary } = useQuery({
    queryKey: ["active-members-summary"],
    queryFn: async () => (await api.get("/reports/active-members-summary")).data.data
    ,
    enabled: Boolean(user) && canReports
  });

  const { data: loanSummary } = useQuery({
    queryKey: ["loan-summary"],
    queryFn: async () => (await api.get("/reports/loan-summary")).data.data
    ,
    enabled: Boolean(user) && canReports
  });

  const { data: monthly } = useQuery({
    queryKey: ["monthly-trends"],
    queryFn: async () => (await api.get("/reports/monthly-trends")).data.data,
    enabled: Boolean(user) && canReports
  });

  const { data: yearly } = useQuery({
    queryKey: ["yearly-trends"],
    queryFn: async () => (await api.get("/reports/yearly-trends")).data.data,
    enabled: Boolean(user) && canReports
  });

  const { data: charity } = useQuery({
    queryKey: ["charity-analysis"],
    queryFn: async () => (await api.get("/reports/charity-analysis")).data.data,
    enabled: Boolean(user) && canReports
  });

  const { data: cashier } = useQuery({
    queryKey: ["cashier-summary"],
    queryFn: async () => (await api.get("/reports/cashier-summary")).data.data,
    enabled: Boolean(user) && canCashier
  });

  const { data: requests } = useQuery({
    queryKey: ["requests-statistics"],
    queryFn: async () => (await api.get("/reports/requests-statistics")).data.data
    ,
    enabled: Boolean(user) && canReports
  });

  const { data: overallSavingsVsLoan } = useQuery({
    queryKey: ["overall-savings-vs-loan"],
    queryFn: async () => (await api.get("/reports/overall-savings-vs-loan")).data.data,
    enabled: Boolean(user) && canReports
  });

  const { data: activeSavingsVsLoan } = useQuery({
    queryKey: ["active-savings-vs-loan"],
    queryFn: async () => (await api.get("/reports/active-savings-vs-loan")).data.data,
    enabled: Boolean(user) && canReports
  });

  const savingsLoanData = savingsLoanView === "overall" ? overallSavingsVsLoan : activeSavingsVsLoan;

  const monthlyOption = useMemo(
    () => ({
      tooltip: { trigger: "axis" },
      legend: { textStyle: { color: "#e2e8f0" } },
      xAxis: { type: "category", data: monthly?.months || [], axisLabel: { color: "#94a3b8" } },
      yAxis: { type: "value", axisLabel: { color: "#94a3b8" } },
      series: [
        { name: "Savings", type: "line", data: monthly?.savings || [], color: "#4cc9f0" },
        { name: "Charity", type: "line", data: monthly?.charity || [], color: "#4895ef" },
        { name: "Loan", type: "line", data: monthly?.loan || [], color: "#f72585" },
        { name: "EMI", type: "line", data: monthly?.emi || [], color: "#e63946" }
      ]
    }),
    [monthly]
  );

  const yearlyOption = useMemo(
    () => ({
      tooltip: { trigger: "axis" },
      legend: { textStyle: { color: "#e2e8f0" } },
      xAxis: { type: "category", data: yearly?.years || [], axisLabel: { color: "#94a3b8" } },
      yAxis: { type: "value", axisLabel: { color: "#94a3b8" } },
      series: [
        { name: "Savings", type: "bar", data: yearly?.savings || [], color: "#4cc9f0" },
        { name: "Charity", type: "bar", data: yearly?.charity || [], color: "#4895ef" },
        { name: "Loan", type: "bar", data: yearly?.loan || [], color: "#f72585" },
        { name: "EMI", type: "bar", data: yearly?.emi || [], color: "#e63946" }
      ]
    }),
    [yearly]
  );

  const charityOption = useMemo(
    () => ({
      tooltip: { trigger: "axis" },
      series: [
        {
          name: "Allotted Amount",
          type: "bar",
          barWidth: "55%",
          data: (charity || []).map((item: any) => Number(item.allotted_amount || 0)),
          itemStyle: { color: "#1677ff" }
        }
      ],
      xAxis: {
        type: "category",
        data: (charity || []).map((item: any) => item.type),
        axisLabel: { color: "#94a3b8", rotate: 20 }
      },
      yAxis: { type: "value", axisLabel: { color: "#94a3b8" } }
    }),
    [charity]
  );

  const savingsLoanOption = useMemo(
    () => ({
      tooltip: { trigger: "item" },
      legend: { textStyle: { color: "#e2e8f0" } },
      series: [
        {
          type: "pie",
          radius: ["35%", "70%"],
          data: [
            { name: "Savings", value: savingsLoanData?.total_savings || 0 },
            { name: "Loans", value: savingsLoanData?.total_loans || 0 }
          ],
          color: ["#4cc9f0", "#f72585"]
        }
      ]
    }),
    [savingsLoanData]
  );

  const requestsTrendOption = useMemo(
    () => ({
      tooltip: { trigger: "axis" },
      legend: { textStyle: { color: "#e2e8f0" } },
      xAxis: {
        type: "category",
        data: (requests?.monthly_data || []).map((item: any) => item.month),
        axisLabel: { color: "#94a3b8" }
      },
      yAxis: { type: "value", axisLabel: { color: "#94a3b8" } },
      series: [
        {
          name: "Loan Requests",
          type: "line",
          data: (requests?.monthly_data || []).map((item: any) => item.loan_requests),
          color: "#4361ee"
        },
        {
          name: "Charity Requests",
          type: "line",
          data: (requests?.monthly_data || []).map((item: any) => item.charity_requests),
          color: "#4895ef"
        },
        {
          name: "Release Requests",
          type: "line",
          data: (requests?.monthly_data || []).map((item: any) => item.release_requests),
          color: "#2ec4b6"
        }
      ]
    }),
    [requests]
  );

  const cashierColumns = [
    { title: "Cashier", dataIndex: "name" },
    {
      title: "Total Credit",
      dataIndex: "total_credit",
      render: (value: number) => formatCurrency(value)
    },
    {
      title: "Total Debit",
      dataIndex: "total_debit",
      render: (value: number) => formatCurrency(value)
    },
    {
      title: "Balance",
      dataIndex: "balance",
      render: (value: number) => formatCurrency(value)
    }
  ];

  if (!canReports) {
    return (
      <Row gutter={[16, 16]}>
        <Col span={24}>
          <Card className="panel" title="Reports">
            <Text type="secondary">You don’t have access to view reports.</Text>
          </Card>
        </Col>
      </Row>
    );
  }

  return (
    <Row gutter={[16, 16]}>
      <Col span={24}>
        {canReports && (
          <Card
            className="panel"
            title={
              <Space size={6}>
                <InfoCircleOutlined style={{ color: "#94a3b8" }} />
                Quick Stats
              </Space>
            }
          >
            <div className="stat-strip">
              <div className="stat-tile">
                <Statistic
                  title={
                    <Space size={6}>
                      <TeamOutlined style={{ color: "#4cc9f0" }} />
                      Active Members
                      <Tooltip title="Active memberships in savings plan">
                        <InfoCircleOutlined style={{ color: "#94a3b8" }} />
                      </Tooltip>
                    </Space>
                  }
                  value={counts?.active_members || 0}
                />
              </div>
              <div className="stat-tile">
                <Statistic
                  title={
                    <Space size={6}>
                      <UserDeleteOutlined style={{ color: "#f72585" }} />
                      Inactive Members
                      <Tooltip title="Inactive memberships">
                        <InfoCircleOutlined style={{ color: "#94a3b8" }} />
                      </Tooltip>
                    </Space>
                  }
                  value={counts?.inactive_members || 0}
                />
              </div>
              <div className="stat-tile">
                <Statistic
                  title={
                    <Space size={6}>
                      <FundOutlined style={{ color: "#4895ef" }} />
                      Active Loans
                      <Tooltip title="Loans with status active">
                        <InfoCircleOutlined style={{ color: "#94a3b8" }} />
                      </Tooltip>
                    </Space>
                  }
                  value={counts?.active_loans || 0}
                />
              </div>
              <div className="stat-tile">
                <Statistic
                  title={
                    <Space size={6}>
                      <HeartOutlined style={{ color: "#e63946" }} />
                      Charity Released
                      <Tooltip title="Total charity funds released">
                        <InfoCircleOutlined style={{ color: "#94a3b8" }} />
                      </Tooltip>
                    </Space>
                  }
                  value={formatCurrency(counts?.total_requested || 0)}
                />
              </div>
            </div>
          </Card>
        )}
      </Col>

      <Col span={24}>
        <Card className="panel" title="All Members Summary">
          <div className="stat-strip">
            <div className="stat-tile">
              <Statistic title="Total Credits" value={formatCurrency(summary?.total_amount || 0)} />
            </div>
            <div className="stat-tile">
              <Statistic title="Total Savings" value={formatCurrency(summary?.total_savings || 0)} />
            </div>
            <div className="stat-tile">
              <Statistic title="Total Charity" value={formatCurrency(summary?.total_charity || 0)} />
              <Text type="secondary">Requested: {formatCurrency(counts?.total_requested || 0)}</Text>
            </div>
            <div className="stat-tile">
              <Statistic title="Total Borrowed" value={formatCurrency(summary?.total_borrowed || 0)} />
            </div>
            <div className="stat-tile">
              <Statistic title="Total Loan Paid" value={formatCurrency(summary?.total_loan_paid || 0)} />
            </div>
          </div>
        </Card>
      </Col>

      <Col span={24}>
        {canReports && (
          <Card className="panel" title="Active Members Summary">
          <Row gutter={[16, 16]}>
            <Col xs={24} md={6}>
              <Statistic title="Total Credit" value={formatCurrency(activeSummary?.total_credit || 0)} />
            </Col>
            <Col xs={24} md={6}>
              <Statistic title="Current Savings" value={formatCurrency(activeSummary?.current_savings || 0)} />
            </Col>
            <Col xs={24} md={6}>
              <Statistic title="Total Charity" value={formatCurrency(activeSummary?.total_charity || 0)} />
              <Text type="secondary">Balance: {formatCurrency(activeSummary?.charity_balance || 0)}</Text>
            </Col>
            <Col xs={24} md={6}>
              <Statistic title="Credit Balance" value={formatCurrency(activeSummary?.credit_balance || 0)} />
            </Col>
          </Row>
          </Card>
        )}
      </Col>

      <Col span={24}>
        {canReports && (
          <Card className="panel" title="Active Loan Summary">
          <Row gutter={[16, 16]}>
            <Col xs={24} md={8}>
              <Statistic title="Current Borrowed" value={formatCurrency(loanSummary?.current_borrowed || 0)} />
            </Col>
            <Col xs={24} md={8}>
              <Statistic title="Current Loan Paid" value={formatCurrency(loanSummary?.current_loan_paid || 0)} />
            </Col>
            <Col xs={24} md={8}>
              <Statistic title="Current Outstanding" value={formatCurrency(loanSummary?.current_outstanding || 0)} />
            </Col>
          </Row>
          </Card>
        )}
      </Col>

      <Col span={24}>
        {canCashier && (
          <Card className="panel" title="Cashier Summary Balance">
            <Table
              dataSource={cashier || []}
              columns={cashierColumns}
              rowKey="id"
              pagination={false}
              scroll={{ x: 720 }}
              rowClassName={(record: any) => (record.id === "grand_total" ? "grand-total-row" : "")}
            />
          </Card>
        )}
      </Col>

      <Col xs={24} md={12}>
        <Card className="panel" title="Charity Analysis">
          <ReactECharts option={charityOption} style={{ height: 320 }} />
        </Card>
      </Col>
      <Col xs={24} md={12}>
        <Card
          className="panel"
          title="Savings vs Loan"
          extra={
            <Segmented
              options={[
                { label: "Overall", value: "overall" },
                { label: "Active", value: "active" }
              ]}
              value={savingsLoanView}
              onChange={(value) => setSavingsLoanView(value as "overall" | "active")}
            />
          }
        >
          <ReactECharts option={savingsLoanOption} style={{ height: 320 }} />
        </Card>
      </Col>

      <Col span={24}>
        <Card className="panel" title="Monthly Trends">
          <ReactECharts option={monthlyOption} style={{ height: 360 }} />
        </Card>
      </Col>
      <Col span={24}>
        <Card className="panel" title="Yearly Trends">
          <ReactECharts option={yearlyOption} style={{ height: 320 }} />
        </Card>
      </Col>

      <Col span={24}>
        <Row gutter={[16, 16]}>
          <Col xs={24} md={12}>
            {canReports && (
              <Card className="panel" title="Requests Summary">
              <Row gutter={[12, 12]}>
                <Col xs={24} md={8}>
                  <Card className="panel" style={{ padding: 14 }}>
                    <Text strong>Loan</Text>
                    <div style={{ marginTop: 8 }}>
                      <Space direction="vertical" size={4}>
                        <Text type="secondary">Pending: {requests?.loan_requests?.pending || 0}</Text>
                        <Text type="secondary">Approved: {requests?.loan_requests?.approved || 0}</Text>
                        <Text type="secondary">Rejected: {requests?.loan_requests?.rejected || 0}</Text>
                        <Text strong>Total: {requests?.loan_requests?.total || 0}</Text>
                      </Space>
                    </div>
                  </Card>
                </Col>
                <Col xs={24} md={8}>
                  <Card className="panel" style={{ padding: 14 }}>
                    <Text strong>Charity</Text>
                    <div style={{ marginTop: 8 }}>
                      <Space direction="vertical" size={4}>
                        <Text type="secondary">Pending: {requests?.charity_requests?.pending || 0}</Text>
                        <Text type="secondary">Approved: {requests?.charity_requests?.approved || 0}</Text>
                        <Text type="secondary">Rejected: {requests?.charity_requests?.rejected || 0}</Text>
                        <Text strong>Total: {requests?.charity_requests?.total || 0}</Text>
                      </Space>
                    </div>
                  </Card>
                </Col>
                <Col xs={24} md={8}>
                  <Card className="panel" style={{ padding: 14 }}>
                    <Text strong>Release</Text>
                    <div style={{ marginTop: 8 }}>
                      <Space direction="vertical" size={4}>
                        <Text type="secondary">Pending: {requests?.release_requests?.pending || 0}</Text>
                        <Text type="secondary">Approved: {requests?.release_requests?.approved || 0}</Text>
                        <Text type="secondary">Rejected: {requests?.release_requests?.rejected || 0}</Text>
                        <Text type="secondary">Released: {requests?.release_requests?.released || 0}</Text>
                        <Text strong>Total: {requests?.release_requests?.total || 0}</Text>
                      </Space>
                    </div>
                  </Card>
                </Col>
              </Row>
              </Card>
            )}
          </Col>
          <Col xs={24} md={12}>
            {canReports && (
              <Card className="panel" title="Monthly Request Trends">
                <ReactECharts option={requestsTrendOption} style={{ height: 320 }} />
              </Card>
            )}
          </Col>
        </Row>
      </Col>
    </Row>
  );
};

export default ReportsPage;
