import { Card, Tabs, Table, Form, Input, Button, Select, message, Row, Col, Alert, Tag } from "antd";
import { useQuery, useQueryClient } from "@tanstack/react-query";
import api from "../api/client";

const RequestsPage = () => {
  const queryClient = useQueryClient();
  const [loanForm] = Form.useForm();
  const [charityForm] = Form.useForm();
  const [releaseForm] = Form.useForm();

  const loanQuery = useQuery({
    queryKey: ["loan-requests-my"],
    queryFn: async () => {
      const response = await api.get("/loan-requests/my");
      return response.data.data;
    }
  });

  const charityQuery = useQuery({
    queryKey: ["charity-requests"],
    queryFn: async () => {
      const response = await api.get("/charity-requests");
      return response.data.data;
    }
  });

  const releaseQuery = useQuery({
    queryKey: ["release-requests"],
    queryFn: async () => {
      const response = await api.get("/release-requests");
      return response.data.data;
    }
  });

  const charityTypesQuery = useQuery({
    queryKey: ["charity-types"],
    queryFn: async () => {
      const response = await api.get("/charity-types");
      return response.data.data;
    }
  });

  const membershipsQuery = useQuery({
    queryKey: ["my-active-memberships"],
    queryFn: async () => {
      const response = await api.get("/memberships/active");
      return response.data.data;
    }
  });

  const hasActiveMembership = (membershipsQuery.data || []).length > 0;

  const onCreateLoan = async (values: any) => {
    try {
      const eligibility = await api.get("/loans/eligibility");
      if (!eligibility.data.data.isEligible) {
        message.warning(eligibility.data.data.message || "Not eligible");
        return;
      }
      await api.post("/loan-requests", values);
      message.success("Loan request submitted");
      loanForm.resetFields();
      queryClient.invalidateQueries({ queryKey: ["loan-requests-my"] });
    } catch (err: any) {
      message.error(err?.response?.data?.message || "Loan request failed");
    }
  };

  const onCreateCharity = async (values: any) => {
    try {
      await api.post("/charity-requests", values);
      message.success("Charity request submitted");
      charityForm.resetFields();
      queryClient.invalidateQueries({ queryKey: ["charity-requests"] });
    } catch (err: any) {
      message.error(err?.response?.data?.message || "Charity request failed");
    }
  };

  const onFetchRelease = async (memberId: string) => {
    if (!memberId) {
      return;
    }
    const response = await api.get(`/release-requests/financial/${memberId}`);
    releaseForm.setFieldsValue(response.data.data);
  };

  const onCreateRelease = async (values: any) => {
    try {
      await api.post("/release-requests", values);
      message.success("Release request submitted");
      releaseForm.resetFields();
      queryClient.invalidateQueries({ queryKey: ["release-requests"] });
    } catch (err: any) {
      message.error(err?.response?.data?.message || "Release request failed");
    }
  };

  return (
    <Card className="panel" title="Requests">
      {!hasActiveMembership && (
        <Alert
          type="warning"
          message="You must have an active savings membership to submit requests."
          style={{ marginBottom: 12 }}
        />
      )}
      <Tabs
        items={[
          {
            key: "loan",
            label: "Loan Requests",
            children: (
              <>
                <Form form={loanForm} layout="vertical" onFinish={onCreateLoan} disabled={!hasActiveMembership}>
                  <Row gutter={16}>
                    <Col xs={24} md={8}>
                      <Form.Item name="amount" label="Amount" rules={[{ required: true }]}>
                        <Input />
                      </Form.Item>
                    </Col>
                    <Col xs={24} md={8}>
                      <Form.Item name="term_months" label="Term Months" rules={[{ required: true }]}>
                        <Input />
                      </Form.Item>
                    </Col>
                    <Col xs={24} md={8}>
                      <Form.Item name="reason" label="Reason" rules={[{ required: true }]}>
                        <Input />
                      </Form.Item>
                    </Col>
                  </Row>
                  <Button type="primary" htmlType="submit" disabled={!hasActiveMembership}>
                    Submit Loan Request
                  </Button>
                </Form>
                <Table
                  style={{ marginTop: 16 }}
                  rowKey="id"
                  dataSource={loanQuery.data || []}
                  loading={loanQuery.isLoading}
                  columns={[
                    { title: "Request ID", dataIndex: "request_id" },
                    { title: "Member", dataIndex: "member_id" },
                    { title: "Amount", dataIndex: "amount" },
                    {
                      title: "Status",
                      dataIndex: "status",
                      render: (value: string) => (
                        <Tag color={value === "released" ? "blue" : value === "approved" ? "green" : value === "rejected" ? "red" : "gold"}>
                          {value}
                        </Tag>
                      )
                    }
                  ]}
                />
              </>
            )
          },
          {
            key: "charity",
            label: "Charity Requests",
            children: (
              <>
                <Form form={charityForm} layout="vertical" onFinish={onCreateCharity} disabled={!hasActiveMembership}>
                  <Row gutter={16}>
                    <Col xs={24} md={8}>
                      <Form.Item name="charity_type_id" label="Charity Type" rules={[{ required: true }]}>
                        <Select
                          options={(charityTypesQuery.data || []).map((c: any) => ({
                            label: c.charity_type,
                            value: c.id
                          }))}
                        />
                      </Form.Item>
                    </Col>
                    <Col xs={24} md={8}>
                      <Form.Item name="beneficiary_name" label="Beneficiary" rules={[{ required: true }]}>
                        <Input />
                      </Form.Item>
                    </Col>
                    <Col xs={24} md={8}>
                      <Form.Item name="amount" label="Amount" rules={[{ required: true }]}>
                        <Input />
                      </Form.Item>
                    </Col>
                  </Row>
                  <Form.Item name="reason" label="Reason" rules={[{ required: true }]}>
                    <Input />
                  </Form.Item>
                  <Button type="primary" htmlType="submit" disabled={!hasActiveMembership}>
                    Submit Charity Request
                  </Button>
                </Form>
                <Table
                  style={{ marginTop: 16 }}
                  rowKey="id"
                  dataSource={charityQuery.data || []}
                  loading={charityQuery.isLoading}
                  columns={[
                    { title: "Charity ID", dataIndex: "charity_id" },
                    { title: "Beneficiary", dataIndex: "beneficiary_name" },
                    { title: "Amount", dataIndex: "amount" },
                    { title: "Status", dataIndex: "status" }
                  ]}
                />
              </>
            )
          },
          {
            key: "release",
            label: "Release Requests",
            children: (
              <>
                <Form form={releaseForm} layout="vertical" onFinish={onCreateRelease} disabled={!hasActiveMembership}>
                  <Row gutter={16}>
                    <Col xs={24} md={8}>
                      <Form.Item name="member_id" label="Member" rules={[{ required: true }]}>
                        <Select
                          options={(membershipsQuery.data || []).map((m: any) => ({
                            label: `${m.member_id} - ${m.member_short_name}`,
                            value: m.member_id
                          }))}
                          onChange={onFetchRelease}
                        />
                      </Form.Item>
                    </Col>
                    <Col xs={24} md={8}>
                      <Form.Item name="current_savings" label="Current Savings" rules={[{ required: true }]}>
                        <Input readOnly />
                      </Form.Item>
                    </Col>
                    <Col xs={24} md={8}>
                      <Form.Item name="outstanding_loan" label="Outstanding Loan" rules={[{ required: true }]}>
                        <Input readOnly />
                      </Form.Item>
                    </Col>
                  </Row>
                  <Row gutter={16}>
                    <Col xs={24} md={8}>
                      <Form.Item name="release_amount" label="Release Amount" rules={[{ required: true }]}>
                        <Input readOnly />
                      </Form.Item>
                    </Col>
                    <Col xs={24} md={16}>
                      <Form.Item name="notes" label="Notes">
                        <Input />
                      </Form.Item>
                    </Col>
                  </Row>
                  <Button type="primary" htmlType="submit" disabled={!hasActiveMembership}>
                    Submit Release Request
                  </Button>
                </Form>
                <Table
                  style={{ marginTop: 16 }}
                  rowKey="id"
                  dataSource={releaseQuery.data || []}
                  loading={releaseQuery.isLoading}
                  columns={[
                    { title: "Request ID", dataIndex: "release_request_id" },
                    { title: "Amount", dataIndex: "release_amount" },
                    { title: "Status", dataIndex: "status" }
                  ]}
                />
              </>
            )
          }
        ]}
      />
    </Card>
  );
};

export default RequestsPage;
