import { Card, Tabs, Table, Input, Space, Tag } from "antd";
import { useMemo, useState } from "react";
import { useQuery } from "@tanstack/react-query";
import api from "../api/client";

const RequestsViewPage = () => {
  const [search, setSearch] = useState("");

  const loanQuery = useQuery({
    queryKey: ["loan-requests-view"],
    queryFn: async () => (await api.get("/loan-requests/view")).data.data
  });

  const charityQuery = useQuery({
    queryKey: ["charity-requests-view"],
    queryFn: async () => (await api.get("/charity-requests/view")).data.data
  });

  const releaseQuery = useQuery({
    queryKey: ["release-requests-view"],
    queryFn: async () => (await api.get("/release-requests/view")).data.data
  });

  const q = search.trim().toLowerCase();

  const loans = useMemo(
    () =>
      (loanQuery.data || []).filter((r: any) => {
        if (!q) return true;
        return (
          String(r.request_id || "").toLowerCase().includes(q) ||
          String(r.member_id || "").toLowerCase().includes(q) ||
          String(r.status || "").toLowerCase().includes(q)
        );
      }),
    [loanQuery.data, q]
  );

  const charity = useMemo(
    () =>
      (charityQuery.data || []).filter((r: any) => {
        if (!q) return true;
        return (
          String(r.charity_id || "").toLowerCase().includes(q) ||
          String(r.member_id || "").toLowerCase().includes(q) ||
          String(r.beneficiary_name || "").toLowerCase().includes(q) ||
          String(r.status || "").toLowerCase().includes(q)
        );
      }),
    [charityQuery.data, q]
  );

  const releases = useMemo(
    () =>
      (releaseQuery.data || []).filter((r: any) => {
        if (!q) return true;
        return (
          String(r.release_request_id || "").toLowerCase().includes(q) ||
          String(r.member_id || "").toLowerCase().includes(q) ||
          String(r.status || "").toLowerCase().includes(q)
        );
      }),
    [releaseQuery.data, q]
  );

  return (
    <Card
      className="panel"
      title="All Requests (View Only)"
      extra={
        <Space>
          <Input.Search placeholder="Search by id / member / status" allowClear onChange={(e) => setSearch(e.target.value)} />
        </Space>
      }
    >
      <Tabs
        items={[
          {
            key: "loan",
            label: `Loan (${loans.length})`,
            children: (
              <Table
                size="small"
                rowKey="id"
                loading={loanQuery.isLoading}
                dataSource={loans}
                pagination={{ pageSize: 20 }}
                columns={[
                  { title: "Request ID", dataIndex: "request_id" },
                  {
                    title: "Member",
                    render: (_: any, r: any) =>
                      `${r.member_id}${r.member_short_name ? ` - ${r.member_short_name}` : ""}${r.user_full_name ? ` (${r.user_full_name})` : ""}`
                  },
                  { title: "Amount", dataIndex: "amount" },
                  { title: "Term", dataIndex: "term_months" },
                  {
                    title: "Status",
                    dataIndex: "status",
                    render: (v: string) => <Tag color={v === "approved" ? "green" : v === "rejected" ? "red" : "gold"}>{v}</Tag>
                  },
                  { title: "Created", dataIndex: "created_at" }
                ]}
              />
            )
          },
          {
            key: "charity",
            label: `Charity (${charity.length})`,
            children: (
              <Table
                size="small"
                rowKey="id"
                loading={charityQuery.isLoading}
                dataSource={charity}
                pagination={{ pageSize: 20 }}
                columns={[
                  { title: "Charity ID", dataIndex: "charity_id" },
                  {
                    title: "Member",
                    render: (_: any, r: any) =>
                      `${r.member_id}${r.member_short_name ? ` - ${r.member_short_name}` : ""}${r.user_full_name ? ` (${r.user_full_name})` : ""}`
                  },
                  { title: "Type", dataIndex: "charity_type" },
                  { title: "Beneficiary", dataIndex: "beneficiary_name" },
                  { title: "Amount", dataIndex: "amount" },
                  {
                    title: "Status",
                    dataIndex: "status",
                    render: (v: string) => <Tag color={v === "released" ? "green" : v === "rejected" ? "red" : v === "approved" ? "blue" : "gold"}>{v}</Tag>
                  },
                  { title: "Created", dataIndex: "created_at" }
                ]}
              />
            )
          },
          {
            key: "release",
            label: `Release (${releases.length})`,
            children: (
              <Table
                size="small"
                rowKey="id"
                loading={releaseQuery.isLoading}
                dataSource={releases}
                pagination={{ pageSize: 20 }}
                columns={[
                  { title: "Release ID", dataIndex: "release_request_id" },
                  {
                    title: "Member",
                    render: (_: any, r: any) =>
                      `${r.member_id}${r.member_short_name ? ` - ${r.member_short_name}` : ""}${r.user_full_name ? ` (${r.user_full_name})` : ""}`
                  },
                  { title: "Amount", dataIndex: "release_amount" },
                  {
                    title: "Status",
                    dataIndex: "status",
                    render: (v: string) => <Tag color={v === "released" ? "green" : v === "rejected" ? "red" : "gold"}>{v}</Tag>
                  },
                  { title: "Created", dataIndex: "created_at" }
                ]}
              />
            )
          }
        ]}
      />
    </Card>
  );
};

export default RequestsViewPage;
