import { Card, Switch, Divider, Tabs, Button, Form, Input, message, Select, Row, Col, Typography, Space, Table, Tag } from "antd";
import { useThemeStore } from "../store/themeStore";
import api from "../api/client";
import { useAuthStore } from "../store/authStore";
import { useQuery } from "@tanstack/react-query";
import { useEffect, useMemo, useState } from "react";
import { can } from "../utils/permissions";

const { Paragraph, Text } = Typography;

const SettingsPage = () => {
  const { mode, toggle } = useThemeStore();
  const user = useAuthStore((s) => s.user);
  const permissions = useAuthStore((s) => s.permissions);
  const isAdmin = user?.access_level === "administrator";

  const [passwordForm] = Form.useForm();
  const [brandingForm] = Form.useForm();
  const [previewTemplate, setPreviewTemplate] = useState<string | undefined>();
  const [previewSubject, setPreviewSubject] = useState<string>("Preview");
  const [previewContext, setPreviewContext] = useState<string>(
    JSON.stringify(
      {
        member_name: "Member Name",
        request_id: "LR1234",
        charity_id: "CR1234",
        loan_id: "LN1234",
        amount: 1000,
        transaction_id: "TX1234",
        status: "approved",
        date: new Date().toISOString().slice(0, 10)
      },
      null,
      2
    )
  );
  const [previewHtml, setPreviewHtml] = useState<string>("");

  const onChangePassword = async (values: any) => {
    try {
      await api.post("/auth/change-password", values);
      message.success("Password updated");
      passwordForm.resetFields();
    } catch (err: any) {
      message.error(err?.response?.data?.message || "Password update failed");
    }
  };

  const settingsQuery = useQuery({
    queryKey: ["admin-settings"],
    enabled: isAdmin,
    queryFn: async () => {
      const response = await api.get("/admin/settings");
      return response.data.data;
    }
  });

  const emailTemplatesQuery = useQuery({
    queryKey: ["email-templates"],
    enabled: isAdmin,
    queryFn: async () => {
      const response = await api.get("/admin/email-templates");
      return response.data.data.templates as string[];
    }
  });

  const emailTemplates = useMemo(() => emailTemplatesQuery.data || [], [emailTemplatesQuery.data]);

  const accessRows = useMemo(() => {
    const keys: Array<{ group: string; key: string; label: string }> = [
      { group: "Core Pages", key: "home.view", label: "Home (view)" },
      { group: "Core Pages", key: "profile.view", label: "Profile (view)" },
      { group: "Core Pages", key: "notifications.view", label: "Notifications (view)" },
      { group: "Core Pages", key: "settings.view", label: "Settings (view)" },
      { group: "Reports", key: "reports.view", label: "Reports (view)" },
      { group: "Reports", key: "cashier_summary.view", label: "Cashier summary (view)" },
      { group: "Credits", key: "credits.view", label: "Credits (view)" },
      { group: "Credits", key: "credits.create", label: "Credits (create)" },
      { group: "Credits", key: "credits.update", label: "Credits (update)" },
      { group: "Credits", key: "credits.delete", label: "Credits (delete)" },
      { group: "Debits", key: "debits.view", label: "Debits (view)" },
      { group: "Debits", key: "debits.create", label: "Debits (create)" },
      { group: "Debits", key: "debits.update", label: "Debits (update)" },
      { group: "Debits", key: "debits.delete", label: "Debits (delete)" },
      { group: "Requests", key: "requests.submit", label: "Requests (submit)" },
      { group: "Requests", key: "requests.view_all", label: "All Requests (view-only)" },
      { group: "Admin/Analysis", key: "admin.overview.view", label: "Admin Overview (view)" },
      { group: "Admin/Analysis", key: "admin.member_savings.view", label: "Member Savings (view)" },
      { group: "Admin/Analysis", key: "admin.cashier_analysis.view", label: "Cashier Analysis (view)" }
    ];

    return keys.map((k) => ({ ...k, id: k.key }));
  }, []);

  const [accessDraft, setAccessDraft] = useState<any>(null);

  useEffect(() => {
    if (!settingsQuery.data?.access?.roles) return;
    setAccessDraft(settingsQuery.data.access.roles);
  }, [settingsQuery.data?.access?.roles]);

  const setRolePerm = (role: string, key: string, value: boolean) => {
    setAccessDraft((prev: any) => {
      const next = { ...(prev || {}) };
      next[role] = { ...(next[role] || {}) };
      next[role][key] = value;
      if (role === "administrator") next[role]["*"] = true;
      return next;
    });
  };

  const saveAccess = async () => {
    try {
      await api.put("/admin/settings", { access: { roles: accessDraft } });
      message.success("Access control saved");
    } catch (err: any) {
      message.error(err?.response?.data?.message || "Save failed");
    }
  };

  useEffect(() => {
    if (!settingsQuery.data) return;
    brandingForm.setFieldsValue({
      app_name: settingsQuery.data?.company?.app_name,
      company_name: settingsQuery.data?.company?.company_name,
      logo_url: settingsQuery.data?.company?.logo_url,
      support_email: settingsQuery.data?.company?.support_email,
      primary_color: settingsQuery.data?.company?.primary_color || "#1f2937",
      footer_text: settingsQuery.data?.company?.footer_text,
      cc_admin: String(settingsQuery.data?.email?.cc_admin ?? true)
    });
  }, [settingsQuery.data, brandingForm]);

  const onSaveBranding = async (values: any) => {
    try {
      await api.put("/admin/settings", {
        company: {
          app_name: values.app_name,
          company_name: values.company_name,
          logo_url: values.logo_url,
          support_email: values.support_email,
          primary_color: values.primary_color,
          footer_text: values.footer_text
        },
        email: {
          cc_admin: values.cc_admin === "true"
        }
      });
      message.success("Settings saved");
    } catch (err: any) {
      message.error(err?.response?.data?.message || "Save failed");
    }
  };

  const onPreview = async () => {
    if (!previewTemplate) {
      message.warning("Select a template");
      return;
    }
    try {
      const response = await api.post("/admin/email-preview", {
        template: previewTemplate,
        subject: previewSubject,
        context: previewContext
      });
      setPreviewHtml(response.data.data.html || "");
    } catch (err: any) {
      message.error(err?.response?.data?.message || "Preview failed");
    }
  };

  return (
    <Card className="panel" title="Application Settings">
      <Tabs
        items={[
          {
            key: "appearance",
            label: "Appearance",
            children: (
              <>
                <div style={{ display: "flex", alignItems: "center", gap: 12 }}>
                  <span>Dark Mode</span>
                  <Switch checked={mode === "dark"} onChange={toggle} />
                </div>
                <Divider />
                <div style={{ color: "#94a3b8" }}>Theme preference is saved locally on this device.</div>
              </>
            )
          },
          {
            key: "security",
            label: "Security",
            children: (
              <Card className="panel" title="Change Password">
                <Form form={passwordForm} layout="vertical" onFinish={onChangePassword}>
                  <Form.Item name="current_password" label="Current Password" rules={[{ required: true }]}>
                    <Input.Password />
                  </Form.Item>
                  <Form.Item
                    name="new_password"
                    label="New Password"
                    rules={[{ required: true }, { min: 6, message: "Minimum 6 characters" }]}
                  >
                    <Input.Password />
                  </Form.Item>
                  <Button type="primary" htmlType="submit">
                    Update Password
                  </Button>
                </Form>
              </Card>
            )
          },
          {
            key: "access",
            label: "Access (Info)",
            children: !isAdmin ? (
              <div style={{ color: "#94a3b8" }}>Only administrators can edit access control.</div>
            ) : (
              <Row gutter={[16, 16]}>
                <Col xs={24} lg={16}>
                  <Card
                    className="panel"
                    title="Role-Based Access Control"
                    extra={
                      <Space>
                        <Button type="primary" onClick={saveAccess} disabled={!accessDraft}>
                          Save
                        </Button>
                      </Space>
                    }
                  >
                    <Table
                      size="small"
                      rowKey="id"
                      pagination={false}
                      dataSource={accessRows}
                      columns={[
                        { title: "Group", dataIndex: "group", width: 140 },
                        { title: "Permission", dataIndex: "label" },
                        {
                          title: "Member",
                          width: 120,
                          render: (_: any, r: any) => (
                            <Switch
                              checked={Boolean(accessDraft?.member?.[r.key])}
                              onChange={(v) => setRolePerm("member", r.key, v)}
                              disabled={!accessDraft}
                            />
                          )
                        },
                        {
                          title: "Editor",
                          width: 120,
                          render: (_: any, r: any) => (
                            <Switch
                              checked={Boolean(accessDraft?.editor?.[r.key])}
                              onChange={(v) => setRolePerm("editor", r.key, v)}
                              disabled={!accessDraft}
                            />
                          )
                        },
                        {
                          title: "Admin",
                          width: 120,
                          render: () => <Switch checked disabled />
                        }
                      ]}
                    />
                  </Card>
                </Col>
                <Col xs={24} lg={8}>
                  <Card className="panel" title="Your Effective Access">
                    <div style={{ color: "#94a3b8", marginBottom: 8 }}>
                      Role: <Tag color="gold">{user?.access_level}</Tag>
                    </div>
                    <div style={{ color: "#94a3b8" }}>
                      Example checks:
                      <div style={{ marginTop: 8 }}>
                        <Tag color={can(permissions, "credits.create") ? "green" : "red"}>credits.create</Tag>
                        <Tag color={can(permissions, "requests.view_all") ? "green" : "red"}>requests.view_all</Tag>
                        <Tag color={can(permissions, "reports.view") ? "green" : "red"}>reports.view</Tag>
                      </div>
                    </div>
                    <Divider />
                    <div style={{ color: "#94a3b8" }}>
                      Notes:
                      <ul style={{ marginTop: 8, paddingLeft: 18 }}>
                        <li>Admin always has full access and cannot be locked out.</li>
                        <li>Backend enforces these permissions for Credits/Debits and Requests view-all endpoints.</li>
                      </ul>
                    </div>
                  </Card>
                </Col>
              </Row>
            )
          },
          {
            key: "email",
            label: "Email (Admin)",
            children: !isAdmin ? (
              <div style={{ color: "#94a3b8" }}>Only administrators can edit email branding and preview templates.</div>
            ) : (
              <Row gutter={[16, 16]}>
                <Col xs={24} lg={12}>
                  <Card className="panel" title="Email Branding">
                    <Form
                      form={brandingForm}
                      layout="vertical"
                      onFinish={onSaveBranding}
                    >
                      <Form.Item name="app_name" label="App Name" rules={[{ required: true }]}>
                        <Input placeholder="Jana ERP" />
                      </Form.Item>
                      <Form.Item name="company_name" label="Company Name">
                        <Input placeholder="Jana" />
                      </Form.Item>
                      <Form.Item name="logo_url" label="Logo URL">
                        <Input placeholder="https://..." />
                      </Form.Item>
                      <Form.Item name="support_email" label="Support Email">
                        <Input placeholder="support@..." />
                      </Form.Item>
                      <Form.Item name="primary_color" label="Header Color (hex)">
                        <Input placeholder="#1f2937" />
                      </Form.Item>
                      <Form.Item name="footer_text" label="Footer Text">
                        <Input.TextArea rows={3} />
                      </Form.Item>
                      <Form.Item name="cc_admin" label="CC Admin">
                        <Select options={[{ label: "Enabled", value: "true" }, { label: "Disabled", value: "false" }]} />
                      </Form.Item>
                      <Button type="primary" htmlType="submit" loading={settingsQuery.isFetching}>
                        Save
                      </Button>
                    </Form>
                    <Divider />
                    <Paragraph type="secondary" style={{ marginBottom: 0 }}>
                      Uses backend templates in `backend/app/templates/emails/`. CC admin uses `MAIL_CC_ADMIN` (if set) and
                      can be enabled/disabled here.
                    </Paragraph>
                  </Card>
                </Col>

                <Col xs={24} lg={12}>
                  <Card className="panel" title="Email Preview">
                    <Space direction="vertical" style={{ width: "100%" }}>
                      <div>
                        <Text type="secondary">Template</Text>
                        <Select
                          style={{ width: "100%" }}
                          placeholder="Select template"
                          value={previewTemplate}
                          onChange={setPreviewTemplate}
                          options={emailTemplates.map((t) => ({ label: t, value: t }))}
                          loading={emailTemplatesQuery.isFetching}
                          showSearch
                        />
                      </div>
                      <div>
                        <Text type="secondary">Subject</Text>
                        <Input value={previewSubject} onChange={(e) => setPreviewSubject(e.target.value)} />
                      </div>
                      <div>
                        <Text type="secondary">Context JSON</Text>
                        <Input.TextArea value={previewContext} onChange={(e) => setPreviewContext(e.target.value)} rows={8} />
                      </div>
                      <Button onClick={onPreview} type="primary">
                        Preview
                      </Button>
                      <div style={{ border: "1px solid #334155", borderRadius: 8, overflow: "hidden" }}>
                        <iframe title="email-preview" style={{ width: "100%", height: 520, border: 0 }} srcDoc={previewHtml} />
                      </div>
                    </Space>
                  </Card>
                </Col>
              </Row>
            )
          }
        ]}
      />
    </Card>
  );
};

export default SettingsPage;
