import { useMemo, useState } from "react";
import { Card, Select, Space, Table, Input, Button, Tabs, Popover, Typography } from "antd";
import { useQuery } from "@tanstack/react-query";
import ReactECharts from "echarts-for-react";
import api from "../../api/client";
import { downloadCsv } from "../../utils/exportCsv";

const { Text } = Typography;

const formatCurrency = (value: number) =>
  new Intl.NumberFormat("en-IN", { minimumFractionDigits: 2, maximumFractionDigits: 2 }).format(value || 0);

const tooltipStyles = {
  wrapper: { width: 480 } as const,
  content: {
    backgroundColor: "#1f2937",
    color: "#f9fafb",
    borderRadius: 6,
    border: "1px solid #4b5563",
    padding: 8
  } as const,
  header: { fontWeight: 600, marginBottom: 6, textAlign: "center", fontSize: 12 } as const,
  summary: {
    fontWeight: 600,
    marginBottom: 8,
    textAlign: "center",
    backgroundColor: "#374151",
    padding: "4px 6px",
    borderRadius: 4,
    fontSize: 11
  } as const,
  tableWrap: { maxHeight: 380, overflowY: "auto" as const } as const,
  table: {
    width: "100%",
    borderCollapse: "collapse" as const,
    tableLayout: "fixed" as const,
    fontSize: 11,
    backgroundColor: "#1f2937",
    color: "#f9fafb"
  } as const,
  th: {
    backgroundColor: "#374151",
    padding: "5px 6px",
    border: "1px solid #4b5563",
    fontWeight: 600,
    textAlign: "left" as const,
    fontSize: 10,
    whiteSpace: "nowrap" as const
  } as const,
  td: {
    padding: "4px 6px",
    border: "1px solid #4b5563",
    textAlign: "left" as const,
    fontSize: 10,
    whiteSpace: "nowrap" as const,
    maxWidth: 160,
    overflow: "hidden",
    textOverflow: "ellipsis"
  } as const
};

const TooltipTable = ({
  headers,
  rows,
  emptyText,
  widths
}: {
  headers: string[];
  rows: Array<Array<string | number>>;
  emptyText: string;
  widths?: Array<number | undefined>;
}) => {
  if (!rows.length) {
    return <div style={{ textAlign: "center", color: "#9ca3af", fontSize: 11 }}>{emptyText}</div>;
  }

  return (
    <div style={tooltipStyles.tableWrap}>
      <table style={tooltipStyles.table}>
        {widths && (
          <colgroup>
            {widths.map((w, idx) => (
              <col key={idx} style={w ? { width: w } : undefined} />
            ))}
          </colgroup>
        )}
        <thead>
          <tr>
            {headers.map((h) => (
              <th key={h} style={tooltipStyles.th}>
                {h}
              </th>
            ))}
          </tr>
        </thead>
        <tbody>
          {rows.map((r, idx) => (
            <tr key={idx}>
              {r.map((cell, cidx) => (
                <td
                  key={cidx}
                  style={{
                    ...tooltipStyles.td,
                    backgroundColor: idx % 2 === 0 ? "#1f2937" : "#374151"
                  }}
                >
                  {cell}
                </td>
              ))}
            </tr>
          ))}
        </tbody>
      </table>
    </div>
  );
};

const CashierDetailPopover = ({
  cashierId,
  type,
  label,
  value
}: {
  cashierId: number;
  type: "current_balance" | "credits" | "debits";
  label: string;
  value: number;
}) => {
  const [open, setOpen] = useState(false);
  const { data, isFetching } = useQuery({
    queryKey: ["cashier-overall-details", cashierId, type],
    queryFn: async () =>
      (await api.get("/reports/cashier-overall-details", { params: { cashier_id: cashierId, type, limit: 15 } })).data.data,
    enabled: open
  });

  const content = (
    <div style={tooltipStyles.wrapper}>
      <div style={tooltipStyles.content}>
        <div style={tooltipStyles.header}>{label}</div>
        <div style={tooltipStyles.summary}>
          Total: {formatCurrency(Number(data?.total_amount || 0))} ({data?.total_count ?? 0} items)
        </div>
        {isFetching ? (
          <div style={{ textAlign: "center", color: "#9ca3af", fontSize: 11 }}>Loading...</div>
        ) : type === "current_balance" ? (
          <TooltipTable
            headers={["Date", "Member", "ID", "Amount"]}
            rows={(data?.details || []).map((d: any) => [
              d.effective_date ? String(d.effective_date).slice(0, 10) : "N/A",
              d.member_name || "N/A",
              d.member_id || "N/A",
              formatCurrency(Number(d.current_balance || 0))
            ])}
            emptyText="No details available"
            widths={[78, 170, 62, 90]}
          />
        ) : (
          <TooltipTable
            headers={["Date", "Member", "ID", "Type", "Amount"]}
            rows={(data?.details || []).map((d: any) => [
              d.date ? String(d.date).slice(0, 10) : "N/A",
              d.member_name || "N/A",
              d.member_id || "N/A",
              d.transaction_type || "N/A",
              formatCurrency(Number(d.amount || 0))
            ])}
            emptyText="No transaction details available"
            widths={[78, 160, 62, 80, 90]}
          />
        )}
      </div>
    </div>
  );

  return (
    <Popover
      open={open}
      onOpenChange={setOpen}
      trigger={["hover", "click"]}
      placement="right"
      content={content}
      overlayStyle={{ zIndex: 2200 }}
      overlayInnerStyle={{ background: "transparent", boxShadow: "none", padding: 0 }}
    >
      <Button type="link" style={{ padding: 0 }}>
        {formatCurrency(value)}
      </Button>
    </Popover>
  );
};

const CashierMonthlyCellPopover = ({
  cashierId,
  cashierName,
  year,
  monthName,
  monthNum,
  type,
  amount,
  count
}: {
  cashierId: number;
  cashierName: string;
  year: number;
  monthName: string;
  monthNum: number;
  type: "credit" | "debit";
  amount: number;
  count: number;
}) => {
  const [open, setOpen] = useState(false);
  const { data, isFetching } = useQuery({
    queryKey: ["cashier-monthly-details", cashierId, year, monthNum, type],
    queryFn: async () =>
      (await api.get("/reports/cashier-monthly-details", { params: { cashier_id: cashierId, year, month: monthNum, type } })).data.data,
    enabled: open
  });

  if (!amount) return <Text type="secondary">-</Text>;

  const title = `${type === "credit" ? "Credits" : "Debits"} - ${monthName} ${year}`;

  const content = (
    <div style={tooltipStyles.wrapper}>
      <div style={tooltipStyles.content}>
        <div style={tooltipStyles.header}>{cashierName}</div>
        <div style={tooltipStyles.header}>{title}</div>
        <div style={tooltipStyles.summary}>
          Total: {formatCurrency(Number(data?.total_amount || 0))} ({data?.total_count ?? 0} transactions)
        </div>
        {isFetching ? (
          <div style={{ textAlign: "center", color: "#9ca3af", fontSize: 11 }}>Loading...</div>
        ) : (
          <TooltipTable
            headers={["Date", "Member", "ID", "Type", "Amount"]}
            rows={(data?.details || []).map((d: any) => [
              d.date ? String(d.date).slice(0, 10) : "N/A",
              d.member_name || "N/A",
              d.member_id || "N/A",
              d.transaction_type || "N/A",
              formatCurrency(Number(d.amount || 0))
            ])}
            emptyText="No transaction details available"
            widths={[78, 160, 62, 80, 90]}
          />
        )}
      </div>
    </div>
  );

  return (
    <Popover
      open={open}
      onOpenChange={setOpen}
      trigger={["hover", "click"]}
      placement="right"
      content={content}
      overlayStyle={{ zIndex: 2200 }}
      overlayInnerStyle={{ background: "transparent", boxShadow: "none", padding: 0 }}
    >
      <div style={{ textAlign: "center" }}>
        <Button
          type="link"
          style={{
            padding: 0,
            fontWeight: 700,
            color: type === "credit" ? "#28a745" : "#dc3545"
          }}
        >
          {Math.round(amount)}
        </Button>
        <div style={{ fontSize: 11, color: "#9ca3af", marginTop: -6 }}>({count})</div>
      </div>
    </Popover>
  );
};

const CashierAnalysisAdminPage = () => {
  const [year, setYear] = useState<number>(new Date().getFullYear());
  const [search, setSearch] = useState("");

  const monthlyQuery = useQuery({
    queryKey: ["cashier-monthly-summary", year],
    queryFn: async () => (await api.get("/reports/cashier-monthly-summary", { params: { year } })).data.data
  });

  const overallQuery = useQuery({
    queryKey: ["cashier-overall-summary"],
    queryFn: async () => (await api.get("/reports/cashier-overall-summary")).data.data
  });

  const months: string[] = monthlyQuery.data?.months || [];
  const monthlyItems: any[] = monthlyQuery.data?.items || [];

  const monthlyFiltered = useMemo(() => {
    const q = search.trim().toLowerCase();
    if (!q) return monthlyItems;
    return monthlyItems.filter((row: any) => String(row.cashier_name || "").toLowerCase().includes(q));
  }, [monthlyItems, search]);

  const overallFiltered = useMemo(() => {
    const q = search.trim().toLowerCase();
    const rows = overallQuery.data?.items || [];
    if (!q) return rows;
    return rows.filter((row: any) => String(row.cashier_name || "").toLowerCase().includes(q));
  }, [overallQuery.data, search]);

  const totals = useMemo(() => {
    return months.map((m) => {
      let credits = 0;
      let debits = 0;
      monthlyFiltered.forEach((c: any) => {
        credits += Number(c.monthly_data?.[m]?.credits?.amount || 0);
        debits += Number(c.monthly_data?.[m]?.debits?.amount || 0);
      });
      return { month: m, credits, debits };
    });
  }, [months, monthlyFiltered]);

  const chartOption = useMemo(
    () => ({
      tooltip: { trigger: "axis" },
      legend: { textStyle: { color: "#e2e8f0" } },
      xAxis: { type: "category", data: totals.map((t) => t.month), axisLabel: { color: "#94a3b8" } },
      yAxis: { type: "value", axisLabel: { color: "#94a3b8" } },
      series: [
        { name: "Credits", type: "bar", data: totals.map((t) => t.credits), color: "#4cc9f0" },
        { name: "Debits", type: "bar", data: totals.map((t) => t.debits), color: "#f72585" }
      ]
    }),
    [totals]
  );

  const monthlyColumns = useMemo(() => {
    return [
      { title: "Cashier", dataIndex: "cashier_name", fixed: "left", width: 160 },
      ...months.map((m) => ({
        title: m,
        children: [
          {
            title: "In",
            dataIndex: ["monthly_data", m, "credits", "amount"],
            width: 60,
            render: (_: any, row: any) => {
              const amount = Number(row.monthly_data?.[m]?.credits?.amount || 0);
              const count = Number(row.monthly_data?.[m]?.credits?.count || 0);
              const monthNum = Number(row.monthly_data?.[m]?.month_num || 0);
              if (!monthNum) return <Text type="secondary">-</Text>;
              return (
                <CashierMonthlyCellPopover
                  cashierId={row.cashier_id}
                  cashierName={row.cashier_name}
                  year={year}
                  monthName={m}
                  monthNum={monthNum}
                  type="credit"
                  amount={amount}
                  count={count}
                />
              );
            }
          },
          {
            title: "Out",
            dataIndex: ["monthly_data", m, "debits", "amount"],
            width: 60,
            render: (_: any, row: any) => {
              const amount = Number(row.monthly_data?.[m]?.debits?.amount || 0);
              const count = Number(row.monthly_data?.[m]?.debits?.count || 0);
              const monthNum = Number(row.monthly_data?.[m]?.month_num || 0);
              if (!monthNum) return <Text type="secondary">-</Text>;
              return (
                <CashierMonthlyCellPopover
                  cashierId={row.cashier_id}
                  cashierName={row.cashier_name}
                  year={year}
                  monthName={m}
                  monthNum={monthNum}
                  type="debit"
                  amount={amount}
                  count={count}
                />
              );
            }
          }
        ]
      }))
    ];
  }, [months, year]);

  const flatCsvRows = useMemo(() => {
    return monthlyFiltered.map((c: any) => {
      const row: any = { cashier_id: c.cashier_id, cashier_name: c.cashier_name, year };
      months.forEach((m) => {
        row[`${m}_credits`] = Number(c.monthly_data?.[m]?.credits?.amount || 0);
        row[`${m}_credits_count`] = Number(c.monthly_data?.[m]?.credits?.count || 0);
        row[`${m}_debits`] = Number(c.monthly_data?.[m]?.debits?.amount || 0);
        row[`${m}_debits_count`] = Number(c.monthly_data?.[m]?.debits?.count || 0);
      });
      return row;
    });
  }, [monthlyFiltered, months, year]);

  const csvColumns = useMemo(() => {
    const cols: any[] = [{ title: "Cashier", dataIndex: "cashier_name" }, { title: "Year", dataIndex: "year" }];
    months.forEach((m) => {
      cols.push({ title: `${m} In`, dataIndex: `${m}_credits` });
      cols.push({ title: `${m} In Count`, dataIndex: `${m}_credits_count` });
      cols.push({ title: `${m} Out`, dataIndex: `${m}_debits` });
      cols.push({ title: `${m} Out Count`, dataIndex: `${m}_debits_count` });
    });
    return cols;
  }, [months]);

  return (
    <Card
      className="panel"
      title="Cashier Summary & Analysis"
      extra={
        <Space wrap>
          <Select
            value={year}
            style={{ width: 120 }}
            onChange={(v) => setYear(v)}
            options={Array.from({ length: 8 }).map((_, i) => {
              const y = new Date().getFullYear() - i;
              return { label: String(y), value: y };
            })}
          />
          <Input.Search
            placeholder="Search cashier"
            allowClear
            onChange={(e) => setSearch(e.target.value)}
            style={{ width: 220 }}
          />
          <Button onClick={() => downloadCsv(`cashier_monthly_${year}.csv`, flatCsvRows, csvColumns)}>Export CSV</Button>
        </Space>
      }
    >
      <Tabs
        items={[
          {
            key: "overall",
            label: "Overall",
            children: (
              <Table
                loading={overallQuery.isLoading}
                dataSource={overallFiltered}
                rowKey="cashier_id"
                pagination={{ pageSize: 20 }}
                scroll={{ x: 1300 }}
                columns={[
                  { title: "Cashier", dataIndex: "cashier_name", fixed: "left", width: 180 },
                  { title: "Active Members", dataIndex: "active_members_count", width: 140 },
                  {
                    title: "Current Balance",
                    dataIndex: "total_current_balance",
                    width: 160,
                    render: (value: any, row: any) => (
                      <CashierDetailPopover
                        cashierId={row.cashier_id}
                        type="current_balance"
                        label="Current Balance"
                        value={Number(value || 0)}
                      />
                    )
                  },
                  {
                    title: "Credits Processed",
                    dataIndex: "total_credits_processed",
                    width: 170,
                    render: (value: any, row: any) => (
                      <CashierDetailPopover
                        cashierId={row.cashier_id}
                        type="credits"
                        label="Credits Processed"
                        value={Number(value || 0)}
                      />
                    )
                  },
                  {
                    title: "Debits Processed",
                    dataIndex: "total_debits_processed",
                    width: 170,
                    render: (value: any, row: any) => (
                      <CashierDetailPopover
                        cashierId={row.cashier_id}
                        type="debits"
                        label="Debits Processed"
                        value={Number(value || 0)}
                      />
                    )
                  },
                  {
                    title: "Net",
                    dataIndex: "net_transactions",
                    width: 140,
                    render: (v: any) => formatCurrency(Number(v || 0))
                  },
                  { title: "Transactions", dataIndex: "total_transactions", width: 140 },
                  {
                    title: "First",
                    dataIndex: "first_transaction",
                    width: 120,
                    render: (v: any) => (v ? String(v).slice(0, 10) : "N/A")
                  },
                  {
                    title: "Last",
                    dataIndex: "last_transaction",
                    width: 120,
                    render: (v: any) => (v ? String(v).slice(0, 10) : "N/A")
                  }
                ]}
              />
            )
          },
          {
            key: "monthly",
            label: "Monthly",
            children: (
              <>
                <Card className="panel" title="Totals Chart" style={{ marginBottom: 16 }}>
                  <ReactECharts option={chartOption} style={{ height: 320 }} />
                </Card>
                <Table
                  loading={monthlyQuery.isLoading}
                  dataSource={monthlyFiltered}
                  rowKey="cashier_id"
                  columns={monthlyColumns as any}
                  pagination={{ pageSize: 20 }}
                  scroll={{ x: 160 + months.length * 120 }}
                />
              </>
            )
          }
        ]}
      />
    </Card>
  );
};

export default CashierAnalysisAdminPage;
