import { useMemo, useState } from "react";
import { Card, Table, Space, Input, Select, Button, Statistic, Row, Col } from "antd";
import { useQuery } from "@tanstack/react-query";
import api from "../../api/client";
import { downloadCsv } from "../../utils/exportCsv";

const MemberSavingsAdminPage = () => {
  const [search, setSearch] = useState("");
  const [status, setStatus] = useState<"active" | "inactive" | "all">("active");
  const [cashierId, setCashierId] = useState<number | undefined>();

  const { data: cashiers } = useQuery({
    queryKey: ["cashiers"],
    queryFn: async () => (await api.get("/cashiers")).data.data
  });

  const { data, isLoading } = useQuery({
    queryKey: ["member-savings-summary", status, cashierId],
    queryFn: async () => {
      const response = await api.get("/reports/member-savings-summary", { params: { status, cashier_id: cashierId } });
      return response.data.data;
    }
  });

  const filtered = useMemo(() => {
    const q = search.trim().toLowerCase();
    const items = data?.items || [];
    if (!q) return items;
    return items.filter((row: any) => {
      return (
        String(row.member_id || "").toLowerCase().includes(q) ||
        String(row.member_short_name || "").toLowerCase().includes(q) ||
        String(row.cashier_name || "").toLowerCase().includes(q)
      );
    });
  }, [data, search]);

  const columns = [
    { title: "Member ID", dataIndex: "member_id" },
    { title: "Short Name", dataIndex: "member_short_name" },
    { title: "Cashier", dataIndex: "cashier_name" },
    { title: "Status", dataIndex: "status" },
    { title: "Savings", dataIndex: "savings" },
    { title: "Charity", dataIndex: "charity" },
    { title: "Loan", dataIndex: "loan" },
    { title: "Loan Paid", dataIndex: "loan_paid" },
    { title: "Outstanding", dataIndex: "outstanding" }
  ];

  return (
    <>
      <Card className="panel" title="Member Savings Summary" style={{ marginBottom: 16 }}>
        <Row gutter={[16, 16]}>
          <Col xs={24} md={6}>
            <Statistic title="Total Savings" value={data?.grand_totals?.savings || 0} />
          </Col>
          <Col xs={24} md={6}>
            <Statistic title="Total Charity" value={data?.grand_totals?.charity || 0} />
          </Col>
          <Col xs={24} md={6}>
            <Statistic title="Total Loan" value={data?.grand_totals?.loan || 0} />
          </Col>
          <Col xs={24} md={6}>
            <Statistic title="Outstanding" value={data?.grand_totals?.outstanding || 0} />
          </Col>
        </Row>
      </Card>

      <Card
        className="panel"
        title="Members"
        extra={
          <Space wrap>
            <Select
              value={status}
              style={{ width: 140 }}
              onChange={(v) => setStatus(v)}
              options={[
                { label: "Active", value: "active" },
                { label: "Inactive", value: "inactive" },
                { label: "All", value: "all" }
              ]}
            />
            <Select
              allowClear
              placeholder="Cashier"
              style={{ width: 180 }}
              value={cashierId}
              onChange={(v) => setCashierId(v)}
              options={(cashiers || []).map((c: any) => ({ label: c.name, value: c.id }))}
            />
            <Input.Search placeholder="Search member" allowClear onChange={(e) => setSearch(e.target.value)} style={{ width: 220 }} />
            <Button onClick={() => downloadCsv("member_savings_summary.csv", filtered, columns as any)}>Export CSV</Button>
          </Space>
        }
      >
        <Table loading={isLoading} dataSource={filtered} rowKey="member_id" columns={columns as any} pagination={{ pageSize: 20 }} />
      </Card>
    </>
  );
};

export default MemberSavingsAdminPage;

