import { useMemo, useState } from "react";
import { Card, Table, Button, Drawer, Form, Input, Select, message, Space } from "antd";
import { useQuery, useQueryClient } from "@tanstack/react-query";
import api from "../../api/client";
import { downloadCsv } from "../../utils/exportCsv";

const MembersAdminPage = () => {
  const queryClient = useQueryClient();
  const [open, setOpen] = useState(false);
  const [editing, setEditing] = useState<any | null>(null);
  const [form] = Form.useForm();
  const [search, setSearch] = useState("");
  const [statusFilter, setStatusFilter] = useState<"active" | "inactive" | undefined>();
  const [cashierFilter, setCashierFilter] = useState<number | undefined>();
  const [planFilter, setPlanFilter] = useState<number | undefined>();

  const { data, isLoading } = useQuery({
    queryKey: ["memberships"],
    queryFn: async () => {
      const response = await api.get("/memberships");
      return response.data.data;
    }
  });

  const { data: users } = useQuery({
    queryKey: ["admin-users"],
    queryFn: async () => {
      const response = await api.get("/admin/users");
      return response.data.data;
    }
  });

  const { data: planTypes } = useQuery({
    queryKey: ["savings-plan-types"],
    queryFn: async () => {
      const response = await api.get("/savings-plan-types");
      return response.data.data;
    }
  });

  const { data: cashiers } = useQuery({
    queryKey: ["cashiers"],
    queryFn: async () => {
      const response = await api.get("/cashiers");
      return response.data.data;
    }
  });

  const userNameById = useMemo(() => {
    const map = new Map<number, string>();
    (users || []).forEach((u: any) => map.set(u.id, u.full_name));
    return map;
  }, [users]);

  const planNameById = useMemo(() => {
    const map = new Map<number, string>();
    (planTypes || []).forEach((p: any) => map.set(p.id, p.name));
    return map;
  }, [planTypes]);

  const cashierNameById = useMemo(() => {
    const map = new Map<number, string>();
    (cashiers || []).forEach((c: any) => map.set(c.id, c.name));
    return map;
  }, [cashiers]);

  const userOptions = useMemo(() => {
    const approvedUsers = (users || []).filter((u: any) => u.account_status === "Approved");
    const activeUserIds = new Set(
      (data || []).filter((m: any) => m.status === "active").map((m: any) => m.user_id)
    );
    const eligibleUsers = approvedUsers.filter((u: any) => !activeUserIds.has(u.id));
    if (editing && editing.user_id) {
      const current = approvedUsers.find((u: any) => u.id === editing.user_id);
      if (current && !eligibleUsers.find((u: any) => u.id === current.id)) {
        eligibleUsers.unshift(current);
      }
    }
    return eligibleUsers.map((u: any) => ({
      label: `${u.full_name} (${u.email})`,
      value: u.id
    }));
  }, [users, data, editing]);

  const onEdit = (record: any) => {
    setEditing(record);
    setOpen(true);
    form.setFieldsValue({
      member_id: record.member_id,
      member_short_name: record.member_short_name,
      user_id: record.user_id,
      plan_type_id: record.plan_type_id,
      cashier_id: record.cashier_id,
      effective_date: record.effective_date,
      end_date: record.end_date,
      status: record.status
    });
  };

  const onSave = async (values: any) => {
    try {
      await api.put(`/memberships/${editing.id}`, values);
      message.success("Member updated");
      setOpen(false);
      setEditing(null);
      form.resetFields();
      queryClient.invalidateQueries({ queryKey: ["memberships"] });
    } catch (err: any) {
      message.error(err?.response?.data?.message || "Save failed");
    }
  };

  const onDelete = async (record: any) => {
    try {
      await api.delete(`/memberships/${record.id}`);
      message.success("Member deleted");
      queryClient.invalidateQueries({ queryKey: ["memberships"] });
    } catch (err: any) {
      message.error(err?.response?.data?.message || "Delete failed");
    }
  };

  const filtered = useMemo(() => {
    const q = search.trim().toLowerCase();
    return (data || []).filter((row: any) => {
      if (statusFilter && row.status !== statusFilter) return false;
      if (cashierFilter && Number(row.cashier_id) !== cashierFilter) return false;
      if (planFilter && Number(row.plan_type_id) !== planFilter) return false;
      if (!q) return true;
      return (
        String(row.member_id || "").toLowerCase().includes(q) ||
        String(row.member_short_name || "").toLowerCase().includes(q) ||
        String(row.user_id || "").toLowerCase().includes(q) ||
        String(row.status || "").toLowerCase().includes(q)
      );
    });
  }, [data, search, statusFilter, cashierFilter, planFilter, cashiers, planTypes]);

  return (
    <Card
      className="panel"
      title="Members"
      extra={
        <Space>
          <Select
            allowClear
            placeholder="Status"
            style={{ width: 140 }}
            value={statusFilter}
            onChange={(v) => setStatusFilter(v)}
            options={[
              { label: "Active", value: "active" },
              { label: "Inactive", value: "inactive" }
            ]}
          />
          <Select
            allowClear
            placeholder="Cashier"
            style={{ width: 180 }}
            value={cashierFilter}
            onChange={(v) => setCashierFilter(v)}
            options={(cashiers || []).map((c: any) => ({ label: c.name, value: c.id }))}
          />
          <Select
            allowClear
            placeholder="Plan"
            style={{ width: 180 }}
            value={planFilter}
            onChange={(v) => setPlanFilter(v)}
            options={(planTypes || []).map((p: any) => ({ label: p.name, value: p.id }))}
          />
          <Input.Search placeholder="Search" allowClear onChange={(e) => setSearch(e.target.value)} style={{ width: 220 }} />
          <Button
            onClick={() =>
              downloadCsv("memberships.csv", filtered, [
                { title: "Member ID", dataIndex: "member_id" },
                { title: "Short Name", dataIndex: "member_short_name" },
                { title: "User ID", dataIndex: "user_id" },
                { title: "Plan Type ID", dataIndex: "plan_type_id" },
                { title: "Cashier ID", dataIndex: "cashier_id" },
                { title: "Status", dataIndex: "status" }
              ])
            }
          >
            Export CSV
          </Button>
        </Space>
      }
    >
      <Table
        loading={isLoading}
        dataSource={filtered}
        rowKey="id"
        columns={[
          { title: "Member ID", dataIndex: "member_id" },
          { title: "Short Name", dataIndex: "member_short_name" },
          {
            title: "User",
            dataIndex: "user_id",
            render: (value: number) => userNameById.get(value) || value
          },
          {
            title: "Plan",
            dataIndex: "plan_type_id",
            render: (value: number) => planNameById.get(value) || value
          },
          {
            title: "Cashier",
            dataIndex: "cashier_id",
            render: (value: number) => cashierNameById.get(value) || value
          },
          { title: "Status", dataIndex: "status" },
          {
            title: "Actions",
            render: (_: any, record: any) => (
              <Space>
                <Button size="small" onClick={() => onEdit(record)}>
                  Edit
                </Button>
                <Button size="small" danger disabled={record.status === "active"} onClick={() => onDelete(record)}>
                  Delete
                </Button>
              </Space>
            )
          }
        ]}
      />

      <Drawer
        title="Edit Member"
        open={open}
        placement="right"
        onClose={() => setOpen(false)}
        width={420}
      >
        <Form form={form} layout="vertical" onFinish={onSave}>
          <Form.Item label="Member ID">
            <Input value={editing?.member_id} readOnly />
          </Form.Item>
          <Form.Item name="member_short_name" label="Short Name" rules={[{ required: true }]}>
            <Input />
          </Form.Item>
          <Form.Item name="user_id" label="User" rules={[{ required: true }]}>
            <Select options={userOptions} />
          </Form.Item>
          <Form.Item name="plan_type_id" label="Plan Type" rules={[{ required: true }]}>
            <Select
              options={(planTypes || []).map((p: any) => ({
                label: p.name,
                value: p.id
              }))}
            />
          </Form.Item>
          <Form.Item name="cashier_id" label="Cashier" rules={[{ required: true }]}>
            <Select
              options={(cashiers || []).map((c: any) => ({
                label: c.name,
                value: c.id
              }))}
            />
          </Form.Item>
          <Form.Item name="effective_date" label="Effective Date" rules={[{ required: true }]}>
            <Input placeholder="YYYY-MM-DD" />
          </Form.Item>
          <Form.Item name="end_date" label="End Date">
            <Input placeholder="YYYY-MM-DD" />
          </Form.Item>
          <Form.Item name="status" label="Status" rules={[{ required: true }]}>
            <Select
              options={[
                { label: "Active", value: "active" },
                { label: "Inactive", value: "inactive" }
              ]}
            />
          </Form.Item>
          <Button type="primary" htmlType="submit" block>
            Save
          </Button>
        </Form>
      </Drawer>
    </Card>
  );
};

export default MembersAdminPage;
