import { useMemo, useState } from "react";
import { Card, Table, Button, Drawer, Form, Input, InputNumber, message, Space } from "antd";
import { PlusOutlined } from "@ant-design/icons";
import { useQuery, useQueryClient } from "@tanstack/react-query";
import api from "../../api/client";
import { useAuthStore } from "../../store/authStore";
import { downloadCsv } from "../../utils/exportCsv";

const SavingsPlanTypesAdminPage = () => {
  const queryClient = useQueryClient();
  const { user } = useAuthStore();
  const [open, setOpen] = useState(false);
  const [editing, setEditing] = useState<any | null>(null);
  const [form] = Form.useForm();
  const [search, setSearch] = useState("");
  const [minPeriodFilter, setMinPeriodFilter] = useState<number | undefined>();

  const { data, isLoading } = useQuery({
    queryKey: ["savings-plan-types"],
    queryFn: async () => {
      const response = await api.get("/savings-plan-types");
      return response.data.data;
    }
  });

  const onEdit = (record: any) => {
    setEditing(record);
    setOpen(true);
    form.setFieldsValue({
      name: record.name,
      min_period: record.min_period,
      monthly_saving_amount: record.monthly_saving_amount,
      charity_amount: record.charity_amount
    });
  };

  const onCreate = () => {
    setEditing(null);
    form.resetFields();
    setOpen(true);
  };

  const onSave = async (values: any) => {
    const payload = Object.fromEntries(
      Object.entries(values).filter(([, value]) => value !== null && value !== undefined)
    );
    try {
      if (editing) {
        await api.put(`/savings-plan-types/${editing.id}`, payload);
        message.success("Savings plan updated");
      } else {
        await api.post("/savings-plan-types", payload);
        message.success("Savings plan created");
      }
      setOpen(false);
      setEditing(null);
      form.resetFields();
      queryClient.invalidateQueries({ queryKey: ["savings-plan-types"] });
    } catch (err: any) {
      message.error(err?.response?.data?.message || "Save failed");
    }
  };

  const onDelete = async (record: any) => {
    try {
      await api.delete(`/savings-plan-types/${record.id}`);
      message.success("Savings plan deleted");
      queryClient.invalidateQueries({ queryKey: ["savings-plan-types"] });
    } catch (err: any) {
      message.error(err?.response?.data?.message || "Delete failed");
    }
  };

  const columns = [
    { title: "Name", dataIndex: "name" },
    { title: "Min Period", dataIndex: "min_period" },
    { title: "Monthly Saving", dataIndex: "monthly_saving_amount" },
    { title: "Charity Amount", dataIndex: "charity_amount" }
  ];

  if (user?.access_level === "administrator") {
    columns.push({
      title: "Actions",
      render: (_: any, record: any) => (
        <Space>
          <Button size="small" onClick={() => onEdit(record)}>
            Edit
          </Button>
          <Button size="small" danger onClick={() => onDelete(record)}>
            Delete
          </Button>
        </Space>
      )
    } as any);
  }

  const filtered = useMemo(() => {
    const q = search.trim().toLowerCase();
    return (data || []).filter((row: any) => {
      if (minPeriodFilter !== undefined && Number(row.min_period || 0) !== minPeriodFilter) return false;
      if (!q) return true;
      return String(row.name || "").toLowerCase().includes(q);
    });
  }, [data, search, minPeriodFilter]);

  return (
    <Card
      className="panel"
      title="Savings Plan Types"
      extra={
        <Space>
          <InputNumber placeholder="Min period" min={0} onChange={(v) => setMinPeriodFilter(v ?? undefined)} />
          <Input.Search placeholder="Search name" allowClear onChange={(e) => setSearch(e.target.value)} style={{ width: 220 }} />
          <Button onClick={() => downloadCsv("savings_plan_types.csv", filtered, columns as any)}>Export CSV</Button>
          {user?.access_level === "administrator" ? (
            <Button type="primary" icon={<PlusOutlined />} onClick={onCreate}>
              New Plan
            </Button>
          ) : null}
        </Space>
      }
    >
      <Table loading={isLoading} dataSource={filtered} rowKey="id" columns={columns} />

      <Drawer
        title={editing ? "Edit Savings Plan" : "Create Savings Plan"}
        open={open}
        placement="right"
        onClose={() => setOpen(false)}
        width={420}
      >
        <Form form={form} layout="vertical" onFinish={onSave}>
          <Form.Item name="name" label="Name" rules={[{ required: true }]}>
            <Input />
          </Form.Item>
          <Form.Item name="min_period" label="Min Period (months)">
            <InputNumber min={0} style={{ width: "100%" }} />
          </Form.Item>
          <Form.Item name="monthly_saving_amount" label="Monthly Saving" rules={[{ required: true }]}>
            <InputNumber min={0} step={0.01} style={{ width: "100%" }} />
          </Form.Item>
          <Form.Item name="charity_amount" label="Charity Amount" rules={[{ required: true }]}>
            <InputNumber min={0} step={0.01} style={{ width: "100%" }} />
          </Form.Item>
          <Button type="primary" htmlType="submit" block>
            Save
          </Button>
        </Form>
      </Drawer>
    </Card>
  );
};

export default SavingsPlanTypesAdminPage;
