import { useMemo, useState } from "react";
import { Card, Table, Tag, Button, Drawer, Form, Select, message, Space, Input, Typography } from "antd";
import { useQuery, useQueryClient } from "@tanstack/react-query";
import api from "../../api/client";
import { downloadCsv } from "../../utils/exportCsv";

const { Text } = Typography;

const UsersAdminPage = () => {
  const queryClient = useQueryClient();
  const [open, setOpen] = useState(false);
  const [editing, setEditing] = useState<any | null>(null);
  const [form] = Form.useForm();
  const [activateOpen, setActivateOpen] = useState(false);
  const [activating, setActivating] = useState<any | null>(null);
  const [activateForm] = Form.useForm();
  const [search, setSearch] = useState("");
  const [statusFilter, setStatusFilter] = useState<"Approved" | "Not Approved" | undefined>();
  const [roleFilter, setRoleFilter] = useState<"administrator" | "editor" | "member" | undefined>();

  const { data, isLoading } = useQuery({
    queryKey: ["admin-users"],
    queryFn: async () => {
      const response = await api.get("/admin/users");
      return response.data.data;
    }
  });

  const { data: memberships } = useQuery({
    queryKey: ["memberships"],
    queryFn: async () => {
      const response = await api.get("/memberships");
      return response.data.data;
    }
  });

  const { data: planTypes } = useQuery({
    queryKey: ["savings-plan-types"],
    queryFn: async () => {
      const response = await api.get("/savings-plan-types");
      return response.data.data;
    }
  });

  const { data: cashiers } = useQuery({
    queryKey: ["cashiers"],
    queryFn: async () => {
      const response = await api.get("/cashiers");
      return response.data.data;
    }
  });

  const activeMembershipMap = useMemo(() => {
    const map = new Map<number, any>();
    (memberships || []).forEach((m: any) => {
      if (m.status === "active") {
        map.set(m.user_id, m);
      }
    });
    return map;
  }, [memberships]);

  const memberIdSuggestion = useMemo(() => {
    const ids = (memberships || []).map((m: any) => String(m.member_id || "")).filter(Boolean);
    const jsNums = ids
      .map((id: string) => {
        const match = id.toUpperCase().match(/^JS(\d+)$/);
        return match ? Number(match[1]) : null;
      })
      .filter((n: any) => typeof n === "number" && Number.isFinite(n)) as number[];
    const maxNum = jsNums.length ? Math.max(...jsNums) : 0;
    return {
      next: `JS${maxNum + 1}`,
      last: maxNum ? `JS${maxNum}` : null
    };
  }, [memberships]);

  const onEdit = (record: any) => {
    setEditing(record);
    setOpen(true);
    form.setFieldsValue({
      account_status: record.account_status,
      access_level: record.access_level
    });
  };

  const onActivate = (record: any) => {
    const today = new Date().toISOString().slice(0, 10);
    setActivating(record);
    setActivateOpen(true);
    activateForm.setFieldsValue({
      user_id: record.id,
      member_id: memberIdSuggestion.next,
      member_short_name: record.full_name,
      effective_date: today,
      status: "active"
    });
  };

  const onSave = async (values: any) => {
    try {
      await api.put(`/admin/users/${editing.id}`, values);
      message.success("User updated");
      setOpen(false);
      setEditing(null);
      form.resetFields();
      queryClient.invalidateQueries({ queryKey: ["admin-users"] });
    } catch (err: any) {
      message.error(err?.response?.data?.message || "Update failed");
    }
  };

  const onActivateSave = async (values: any) => {
    try {
      await api.post("/memberships", values);
      message.success("Membership created");
      setActivateOpen(false);
      setActivating(null);
      activateForm.resetFields();
      queryClient.invalidateQueries({ queryKey: ["memberships"] });
    } catch (err: any) {
      message.error(err?.response?.data?.message || "Activation failed");
    }
  };

  const filtered = useMemo(() => {
    const q = search.trim().toLowerCase();
    return (data || []).filter((row: any) => {
      if (statusFilter && row.account_status !== statusFilter) return false;
      if (roleFilter && row.access_level !== roleFilter) return false;
      if (!q) return true;
      return (
        String(row.username || "").toLowerCase().includes(q) ||
        String(row.email || "").toLowerCase().includes(q) ||
        String(row.full_name || "").toLowerCase().includes(q) ||
        String(row.mobile_number || "").toLowerCase().includes(q) ||
        String(row.access_level || "").toLowerCase().includes(q) ||
        String(row.account_status || "").toLowerCase().includes(q)
      );
    });
  }, [data, search]);

  return (
    <Card
      className="panel"
      title="Users"
      extra={
        <Space>
          <Select
            allowClear
            placeholder="Status"
            style={{ width: 140 }}
            value={statusFilter}
            onChange={(v) => setStatusFilter(v)}
            options={[
              { label: "Approved", value: "Approved" },
              { label: "Not Approved", value: "Not Approved" }
            ]}
          />
          <Select
            allowClear
            placeholder="Role"
            style={{ width: 160 }}
            value={roleFilter}
            onChange={(v) => setRoleFilter(v)}
            options={[
              { label: "Administrator", value: "administrator" },
              { label: "Editor", value: "editor" },
              { label: "Member", value: "member" }
            ]}
          />
          <Input.Search placeholder="Search" allowClear onChange={(e) => setSearch(e.target.value)} style={{ width: 220 }} />
          <Button
            onClick={() =>
              downloadCsv("users.csv", filtered, [
                { title: "Username", dataIndex: "username" },
                { title: "Email", dataIndex: "email" },
                { title: "Full Name", dataIndex: "full_name" },
                { title: "Mobile", dataIndex: "mobile_number" },
                { title: "Role", dataIndex: "access_level" },
                { title: "Account Status", dataIndex: "account_status" }
              ])
            }
          >
            Export CSV
          </Button>
        </Space>
      }
    >
      <Table
        loading={isLoading}
        dataSource={filtered}
        rowKey="id"
        columns={[
          { title: "Username", dataIndex: "username" },
          { title: "Email", dataIndex: "email" },
          { title: "Full Name", dataIndex: "full_name" },
          { title: "Mobile", dataIndex: "mobile_number" },
          { title: "Role", dataIndex: "access_level" },
          {
            title: "Status",
            dataIndex: "account_status",
            render: (value) => <Tag color={value === "Approved" ? "green" : "red"}>{value}</Tag>
          },
          {
            title: "Membership",
            render: (_: any, record: any) => {
              const membership = activeMembershipMap.get(record.id);
              return membership ? (
                <Tag color="green">Active</Tag>
              ) : (
                <Tag color="default">None</Tag>
              );
            }
          },
          {
            title: "Actions",
            render: (_: any, record: any) => (
              <Space>
                <Button size="small" onClick={() => onEdit(record)}>
                  Edit
                </Button>
                <Button
                  size="small"
                  disabled={record.account_status !== "Approved" || activeMembershipMap.has(record.id)}
                  onClick={() => onActivate(record)}
                >
                  Activate Member
                </Button>
              </Space>
            )
          }
        ]}
      />

      <Drawer
        title="Edit User"
        open={open}
        placement="right"
        onClose={() => setOpen(false)}
        width={420}
      >
        <Form form={form} layout="vertical" onFinish={onSave}>
          <Form.Item name="account_status" label="Account Status" rules={[{ required: true }]}>
            <Select
              options={[
                { label: "Approved", value: "Approved" },
                { label: "Not Approved", value: "Not Approved" }
              ]}
            />
          </Form.Item>
          <Form.Item name="access_level" label="Access Level" rules={[{ required: true }]}>
            <Select
              options={[
                { label: "Administrator", value: "administrator" },
                { label: "Editor", value: "editor" },
                { label: "Member", value: "member" }
              ]}
            />
          </Form.Item>
          <Button type="primary" htmlType="submit" block>
            Save
          </Button>
        </Form>
      </Drawer>

      <Drawer
        title={activating ? `Activate ${activating.full_name}` : "Activate Member"}
        open={activateOpen}
        placement="right"
        onClose={() => setActivateOpen(false)}
        width={420}
      >
        {activating && (
          <Form form={activateForm} layout="vertical" onFinish={onActivateSave}>
            <Form.Item name="user_id" label="User ID" rules={[{ required: true }]}>
              <Input readOnly />
            </Form.Item>
            <Form.Item name="member_id" label="Member ID" rules={[{ required: true }]}>
              <Input placeholder="Member ID" />
            </Form.Item>
            <div style={{ marginTop: -8, marginBottom: 16 }}>
              <Text type="secondary">
                Suggested: {memberIdSuggestion.next}
                {memberIdSuggestion.last ? ` (Last used: ${memberIdSuggestion.last})` : ""}
              </Text>
            </div>
            <Form.Item name="member_short_name" label="Short Name" rules={[{ required: true }]}>
              <Input />
            </Form.Item>
            <Form.Item name="plan_type_id" label="Plan Type" rules={[{ required: true }]}>
              <Select
                options={(planTypes || []).map((p: any) => ({
                  label: p.name,
                  value: p.id
                }))}
              />
            </Form.Item>
            <Form.Item name="cashier_id" label="Cashier" rules={[{ required: true }]}>
              <Select
                options={(cashiers || []).map((c: any) => ({
                  label: c.name,
                  value: c.id
                }))}
              />
            </Form.Item>
            <Form.Item name="effective_date" label="Effective Date" rules={[{ required: true }]}>
              <Input placeholder="YYYY-MM-DD" />
            </Form.Item>
            <Form.Item name="end_date" label="End Date">
              <Input placeholder="YYYY-MM-DD" />
            </Form.Item>
            <Form.Item name="status" label="Status" initialValue="active">
              <Select
                options={[
                  { label: "Active", value: "active" },
                  { label: "Inactive", value: "inactive" }
                ]}
              />
            </Form.Item>
            <Button type="primary" htmlType="submit" block>
              Activate
            </Button>
          </Form>
        )}
      </Drawer>
    </Card>
  );
};

export default UsersAdminPage;
